#include <stdio.h>
#include <stdlib.h>

#include <modeller.h>

/* Example of using Modeller from a C program. This simply reads in a PDB
 * file, prints out some data from that file, and then writes out a new
 * file in MMCIF format.
 *
 * To compile, use (where XXX is your Modeller version):
 * gcc -Wall -o c-example c-example.c `modXXX --cflags --libs`
 * (If you use a compiler other than gcc, or a non-Unix system, you may need
 * to run 'modXXX --cflags --libs' manually and construct suitable compiler
 * options by hand.)
 *
 * To run, you must ensure that the Modeller dynamic libraries are in your
 * search path. This can be done on most systems by adding the directory
 * reported by 'modXXX --libs' to the LD_LIBRARY_PATH environment variable.
 * (On Mac, set DYLD_LIBRARY_PATH instead. On Windows, PATH. On AIX, LIBPATH.)
 *
 * You must also ensure that Modeller knows where it was installed,
 * and what the license key is. You can either do this by setting the
 * MODINSTALLXXX and KEY_MODELLERXXX environment variables accordingly, or
 * by calling the set_install_dir() and set_license_key() functions before
 * you call start_modeller(). For example, if Modeller is installed in
 * /lib/modeller on a 32-bit Linux system, the following would work from the
 * command line (all on one line), where KEY is your license key:
 *     KEY_MODELLERXXX=KEY MODINSTALLXXX=/lib/modeller/
 *     LD_LIBRARY_PATH=/lib/modeller/lib/i386-intel8 ./c-example
 */


/* Exit, reporting the Modeller error, iff one occurred. */
void handle_error(int ierr) {
  if (ierr != 0) {
    char *errstr = get_mod_error();
    fprintf(stderr, "Modeller error: %s\n", errstr);
    free(errstr);
    exit(1);
  }
}

int main(void)
{
  struct libraries *libs;
  struct model *mdl;
  struct io_data *io;
  int ierr, *sel1, nsel1;

  /* Uncomment these lines to hard code install location and license key,
     rather than setting MODINSTALLXXX and KEY_MODELLERXXX environment
     variables (see above) */
  /* set_install_dir("/lib/modeller"); */
  /* set_license_key("KEY"); */

  start_modeller(&ierr);
  handle_error(ierr);
  write_header();

  modset_log(2, 1);
  libs = new_libraries(NULL);
  libraries_read_libs(libs, "${LIB}/restyp.lib", -8123, &ierr);
  handle_error(ierr);

  mdl = new_model(NULL);
  io = new_io_data();
  read_model(mdl, io, libs, "../atom_files/1nbt.pdb", "PDB", "FIRST:@LAST:  ",
             7, &ierr);
  handle_error(ierr);
  printf("Model of %s solved at resolution %f, rfactor %f\n", mdl->seq.name,
         mdl->seq.resol, mdl->seq.rfactr);
  select_all(mdl, &sel1, &nsel1);
  write_model(mdl, libs, sel1, nsel1, "new.cif", "MMCIF", 0, 1, &ierr);
  free(sel1);
  handle_error(ierr);
  free_libraries(libs);
  free_model(mdl);
  free_io_data(io);

  end_modeller();
  return 0;
}
