/** \file mod_dynmem.h     Dynamic memory allocation and variable-sized strings.
 *
 *             Part of MODELLER, Copyright(c) 1989-2007 Andrej Sali
 */

#ifndef __MOD_DYNMEM_H
#define __MOD_DYNMEM_H

#include <stdlib.h>
#include <stdarg.h>
#include "dstr_type.h"
#include "mod_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/** A malloc replacement which handles out-of-memory errors */
void *dmalloc(size_t size);

/** A realloc replacement which handles out-of-memory errors */
void *drealloc(void *ptr, size_t size);

/** A strdup replacement which handles out-of-memory errors */
char *dstrdup(const char *str);

/** A strndup replacement which handles out-of-memory errors */
char *dstrndup(const char *str, size_t n);

/** Do a dstrndup, but discard any trailing whitespace */
char *dstrndup_trim(const char *str, size_t n);

/** Create a new dynamic string (dstr) object */
dstr *dstr_new(void);

/** Free the dynamic string object */
void dstr_free(dstr *str);

/** Convert a dynamic string to an ordinary char * buffer. The original
    dynamic string is destroyed. */
char *dstr_to_string(dstr *str);

/** Expand the dynamic string's buffer by extralength bytes */
void dstr_expandby(dstr *str, unsigned extralength);

/** Make sure that the dynamic string is long enough to store a string of
    length bytes */
void dstr_ensurelength(dstr *str, unsigned length);

/** Assign a null-terminated string to the dynamic string */
void dstr_assign(dstr *str, const char *text);

/** Assign at most len bytes of a null-terminated string to the dynamic
    string */
void dstr_assignn(dstr *str, const char *text, int len);

/** Append at most len bytes of a null-terminated string to the dynamic
    string */
void dstr_appendn(dstr *str, const char *text, int len);

/** Append a null-terminated string to the dynamic string */
void dstr_append(dstr *str, const char *text);

/** Append a path component to the string, inserting a separator if needed */
void dstr_append_path(dstr *str, const char *text);

/** Append a single character to the dynamic string */
void dstr_append_c(dstr *str, char ch);

/** Prepend a null-terminated string to the dynamic string */
void dstr_prepend(dstr *str, const char *text);

/** Truncate the string to the requested length */
void dstr_truncate(dstr *str, int len);

/** Delete len bytes from the string, starting at pos. The gap is then closed
    up by moving back anything after it. */
void dstr_erase(dstr *str, int pos, int len);

/** Insert the given text into the string at the given position */
void dstr_insert(dstr *str, int pos, const char *text);

/** Globally substitute orig with newtext */
void dstr_strsubg(dstr *str, const char *orig, const char *newtext);

/** Print into the dynamic string using the given format and va_list (similar
    to vsprintf) */
void dstr_vprintf(dstr *str, const char *format, va_list va);

/** Print into the dynamic string using the given format and arguments (similar
    to sprintf) */
void dstr_printf(dstr *str, const char *format, ...);

/** Print using the given format and arguments (similar to sprintf) into
    a new buffer, which is returned (and should be freed when no longer
    needed.) */
char *dstrdup_printf(const char *format, ...);

/** Copy the C null-terminated string into a Fortran blank-padded buffer.
    If len_to_copy is positive, copy only that many characters from the
    C string. */
void cstring_to_fortran(const char *cstring, int len_to_copy, char *fstring,
                        int_str fstring_len);

/** Split the given string into tokens split by characters in delim */
char **str_split_tokens(const char *str, const char *delim);

/** Free a vector of strings */
void str_freev(char **strvec);

/** Return TRUE iff the dstr ends with the given string */
mbool dstr_endswith(dstr *str, const char *endtext);

#ifdef __cplusplus
}
#endif
#endif   /* __MOD_DYNMEM_H */
