"""Classes for handling template structures in alignments"""

import _modeller
from modeller import sequence, alnsequence, coordinates, modfile

__docformat__ = "epytext en"

class Residue(coordinates.Residue, alnsequence.Residue):
    """A single residue in a template structure."""
    pass


class Structure(coordinates.Coordinates, alnsequence.Sequence):
    """A single template structure within an alignment"""

    __num = None
    __read_coord = False
    _residue_class = Residue

    def __init__(self, aln, num):
        self.__num = num
        alnsequence.Sequence.__init__(self, aln, num)
        coordinates.Coordinates.__init__(self)

    def __repr__(self):
        return "Structure %s" % repr(self.code)
    def __str__(self):
        return "<%s>" % repr(self)

    def write(self, file):
        """Write current coordinates to a PDB file"""
        if isinstance(file, str):
            file = modfile.File(file, 'w')
        return _modeller.mod_alnstructure_write(self.cdpt, self.seqpt,
                                                file.file_pointer,
                                                self.aln.env.libs.modpt)

    def reread(self):
        """Reread coordinates from the atom file. Useful for restoring the
           original template orientation."""
        self.__read_coord = False
        _modeller.mod_alnstructure_invalidate(self.aln.modpt, self.__num)
        self.__get_cdpt()   # Force reread of structural info

    def __get_cdpt(self):
        if not self.__read_coord:
            aln = self.aln
            _modeller.mod_alnstructure_read(aln.modpt, self.__num,
                                            aln.env.io.modpt,
                                            aln.env.libs.modpt)
            self.__read_coord = True
        struc = _modeller.mod_alignment_structure_get(self.aln.modpt,
                                                      self.__num)
        return _modeller.mod_structure_cd_get(struc)
    cdpt = property(__get_cdpt)
