import _modeller
import model
import profile
import util.top as top
import util.modutil as modutil
from modeller.util.modobject import modobject

class alignment(modobject):
    """Holds an alignment of protein sequences"""

    def __init__(self, env, prf=None, io=None, libs=None, **vars):
        self.add_members(('_alignment__modpt', 'env', 'top'))
        self.__modpt = _modeller.new_alignment()
        self.env = env.copy()
        self.top = top.top(self.env)
        if prf is not None:
            self.append_profile(prf, libs)
        elif io is not None or libs is not None or len(vars) > 0:
            self.__int_append('alignment.append', io, libs, vars)

    def __del__(self):
        _modeller.free_alignment(self.modpt)

    def __get_modpt(self):
        return self.__modpt

    def __len__(self):
        return _modeller.get_alignment_nseq(self.modpt)

    def clear(self):
        """Remove all sequences from the alignment"""
        return _modeller.delete_alignment(self.modpt)

    def append(self, io=None, libs=None, **vars):
        """Add sequence(s) from an alignment file"""
        return self.__int_append('alignment.append', io, libs, vars)

    def read(self, io=None, libs=None, **vars):
        """Read sequence(s) from an alignment file"""
        self.clear()
        return self.append(io, libs, **vars)

    def __int_append(self, logname, io, libs, vars):
        if io is None:
            io = self.env.io
        if libs is None:
            libs = self.env.libs
        try:
            align_codes = vars['align_codes']
        except KeyError:
            align_codes = 'all'
        try:
            atom_files = vars['atom_files']
        except KeyError:
            atom_files = None
        vars['align_codes'] = [seq.code for seq in self]
        vars['atom_files'] = [seq.atom_file for seq in self]
        if align_codes is not None:
            if type(align_codes) is tuple: align_codes = list(align_codes)
            if type(align_codes) is not list: align_codes = [ align_codes ]
            vars['align_codes'] = vars['align_codes'] + align_codes
        if atom_files is not None:
            if type(atom_files) is tuple: atom_files = list(atom_files)
            if type(atom_files) is not list: atom_files = [ atom_files ]
            vars['atom_files'] = vars['atom_files'] + atom_files
        vars['add_sequence'] = True
        return self.top.read_alignment(logname, aln=self.modpt, io=io.modpt,
                                       libs=libs.modpt, **vars)

    def append_model(self, mdl, align_codes, atom_files='', libs=None, **vars):
        """Add a sequence from a model"""
        if libs is None:
            libs = self.env.libs
        align_codes = [seq.code for seq in self] + [ align_codes ]
        atom_files = [seq.atom_file for seq in self] + [ atom_files ]
        vars['align_codes'] = align_codes
        vars['atom_files'] = atom_files
        vars['add_sequence'] = True
        return self.top.sequence_to_ali('alignment.append_model',
                                        aln=self.modpt, mdl=mdl.modpt,
                                        libs=libs.modpt, **vars)

    def compare_with(self, aln, **vars):
        """Compare with another alignment"""
        return self.top.compare_alignments('alignment.compare_with',
                                           aln=self.modpt, aln2=aln.modpt,
                                           **vars)

    def compare_structures(self, edat=None, io=None, libs=None, **vars):
        """Compare 3D structures"""
        if io is None:
            io = self.env.io
        if edat is None:
            edat = self.env.edat
        if libs is None:
            libs = self.env.libs
        return self.top.compare('alignment.compare_structures', aln=self.modpt,
                                edat=edat.modpt, io=io.modpt, libs=libs.modpt,
                                **vars)

    def compare_sequences(self, mdl, libs=None, **vars):
        """Compare sequences in alignment"""
        if libs is None:
            libs = self.env.libs
        return self.top.sequence_comparison('alignment.compare_sequences',
                                            aln=self.modpt, mdl=mdl.modpt,
                                            libs=libs.modpt, **vars)

    def segment_matching(self, libs=None, **vars):
        """Align segments"""
        if libs is None:
            libs = self.env.libs
        return self.top.segment_matching('alignment.segment_matching',
                                         aln=self.modpt, libs=libs.modpt,
                                         **vars)

    def edit(self, io=None, libs=None, **vars):
        """Edit overhangs in alignment"""
        if io is None:
            io = self.env.io
        if libs is None:
            libs = self.env.libs
        return self.top.edit_alignment('alignment.edit', aln=self.modpt,
                                       io=io.modpt, libs=libs.modpt, **vars)

    def append_profile(self, prf, libs=None):
        """Add sequences from a profile"""
        if libs is None:
            libs = self.env.libs
        return self.top.prof_to_aln('alignment.append_profile', prf=prf.modpt,
                                    aln=self.modpt, libs=libs.modpt,
                                    append_aln=True)

    def to_profile(self, **vars):
        """Converts the alignment to profile format"""
        prf = profile.profile(self.env)
        self.top.aln_to_prof('alignment.to_profile', aln=self.modpt,
                             prf=prf.modpt, **vars)
        return prf

    def expand(self, **vars):
        """Put all models into alignment"""
        return self.top.expand_alignment('alignment.expand', aln=self.modpt,
                                         **vars)

    def check(self, io=None, libs=None, **vars):
        """Check alignment for modeling"""
        if io is None:
            io = self.env.io
        if libs is None:
            libs = self.env.libs
        return self.top.check_alignment('alignment.check', aln=self.modpt,
                                        io=io.modpt, libs=libs.modpt, **vars)

    def describe(self, io=None, libs=None, **vars):
        """Describe proteins"""
        if io is None:
            io = self.env.io
        if libs is None:
            libs = self.env.libs
        return self.top.describe('alignment.describe', aln=self.modpt,
                                 io=io.modpt, libs=libs.modpt, **vars)

    def consensus(self, libs=None, **vars):
        """Produce a consensus alignment"""
        if libs is None:
            libs = self.env.libs
        return self.top.align_consensus('alignment.consensus', aln=self.modpt,
                                        libs=libs.modpt, **vars)

    def align(self, libs=None, **vars):
        """Align two (blocks of) sequences"""
        if libs is None:
            libs = self.env.libs
        return self.top.align('alignment.align', aln=self.modpt,
                              libs=libs.modpt, **vars)

    def malign(self, libs=None, **vars):
        """Align two or more sequences"""
        if libs is None:
            libs = self.env.libs
        return self.top.malign('alignment.malign', aln=self.modpt,
                               libs=libs.modpt, **vars)

    def align2d(self, io=None, libs=None, **vars):
        """Align sequences with structures"""
        if io is None:
            io = self.env.io
        if libs is None:
            libs = self.env.libs
        return self.top.align2d('alignment.align2d', aln=self.modpt,
                                io=io.modpt, libs=libs.modpt, **vars)

    def align3d(self, io=None, libs=None, **vars):
        """Align two structures"""
        if io is None:
            io = self.env.io
        if libs is None:
            libs = self.env.libs
        return self.top.align3d('alignment.align3d', aln=self.modpt,
                                io=io.modpt, libs=libs.modpt, **vars)

    def malign3d(self, io=None, libs=None, **vars):
        """Align two or more structures"""
        if io is None:
            io = self.env.io
        if libs is None:
            libs = self.env.libs
        return self.top.malign3d('alignment.malign3d', aln=self.modpt,
                                 io=io.modpt, libs=libs.modpt, **vars)

    def salign(self, io=None, libs=None, **vars):
        """Align two or more proteins"""
        if io is None:
            io = self.env.io
        if libs is None:
            libs = self.env.libs
        return self.top.salign('alignment.salign', aln=self.modpt, io=io.modpt,
                               libs=libs.modpt, **vars)

    def write(self, file, libs=None, **vars):
        """Write the alignment to a file"""
        if libs is None:
            libs = self.env.libs
        vars['file'] = file
        return self.top.write_alignment('alignment.write', aln=self.modpt,
                                        libs=libs.modpt, **vars)

    def id_table(self, libs=None, **vars):
        """Calculate percentage sequence identities"""
        if libs is None:
            libs = self.env.libs
        return self.top.id_table('alignment.id_table', aln=self.modpt,
                                 libs=libs.modpt, **vars)

    def __getitem__(self, indx):
        ret = modutil.handle_seq_indx(self, indx)
        if type(ret) is int:
            return sequence(self, ret)
        else:
            return ret
    modpt = property(__get_modpt)


class sequence(object):
    """A single sequence within an alignment"""

    def __init__(self, aln, num):
        self.aln = aln
        self.num = num

    def __len__(self):
        return _modeller.get_alignment_naln(self.aln.modpt)

    def __get_nres(self):
        return _modeller.get_alignment_nresn(self.aln.modpt, self.num)
    def __get_code(self):
        return _modeller.get_alignment_codes(self.aln.modpt, self.num)
    def __set_code(self, val):
        _modeller.set_alignment_codes(self.aln.modpt, self.num, val)
    def __get_atom_file(self):
        return _modeller.get_alignment_atom_files(self.aln.modpt, self.num)
    def __set_atom_file(self, val):
        _modeller.set_alignment_atom_files(self.aln.modpt, self.num, val)
    def __get_resol(self):
        return _modeller.get_alignment_resol(self.aln.modpt, self.num)
    def __set_resol(self, val):
        _modeller.set_alignment_resol(self.aln.modpt, self.num, val)
    def __get_source(self):
        return _modeller.get_alignment_source(self.aln.modpt, self.num)
    def __set_source(self, val):
        _modeller.set_alignment_source(self.aln.modpt, self.num, val)
    def __get_name(self):
        return _modeller.get_alignment_name(self.aln.modpt, self.num)
    def __set_name(self, val):
        _modeller.set_alignment_name(self.aln.modpt, self.num, val)
    def __get_residues(self):
        return residuelist(self)

    code = property(__get_code, __set_code)
    atom_file = property(__get_atom_file, __set_atom_file)
    resol = property(__get_resol, __set_resol)
    source = property(__get_source, __set_source)
    name = property(__get_name, __set_name)
    nres = property(__get_nres)
    residues = property(__get_residues)


class residuelist(object):

    def __init__(self, seq, offset=0, length=None):
        self.seq = seq
        self.offset = offset
        self.length = length

    def __len__(self):
        if self.length is not None:
            return self.length
        else:
            return len(self.seq)

    def __getitem__(self, indx):
        ret = modutil.handle_seq_indx(self, indx)
        if type(ret) is int:
            return residue(self.seq, ret + self.offset)
        else:
            return ret


class residue(object):

    def __init__(self, seq, num):
        self.seq = seq
        self.num = num

    def __get_name(self):
        return _modeller.get_alignment_caln(self.seq.aln.modpt, self.seq.num,
                                            self.num)

    name = property(__get_name)
