"""Classes to get normalized DOPE scores."""


class dope_scorer(object):
    """Use raw DOPE scores to get normalized measures."""

    aa_scores = { 'ALA': -78.2782,
                  'ARG': -105.292,
                  'ASN': -53.1271,
                  'ASP': -56.2418,
                  'CYS': -221.908,
                  'GLN': -95.1577,
                  'GLU': -89.5431,
                  'GLY': -67.8921,
                  'HIS': -110.174,
                  'ILE': -122.583,
                  'LEU': -144.657,
                  'LYS': -102.768,
                  'MET': -91.2464,
                  'PHE': -178.048,
                  'PRO': -50.2028,
                  'SER': -85.3045,
                  'THR': -61.2169,
                  'TRP': -240.059,
                  'TYR': -149.113,
                  'VAL': -113.097
                }

    def __init__(self, mdl):
        self.__mdl = mdl


    def get_z_score(self, value):
        """Given a DOPE score, return a normalized score (z value)"""
        (mean, numstdres) = self._get_mean(self.__mdl)
        return (value - mean) / self._get_stdev(numstdres)


    def get_profile(self, prof):
        """Given a raw DOPE profile, return a normalized profile"""
        from modeller.energy_profile import energy_profile
        mdl = self.__mdl
        numstdres = 0
        norm_prof = [0.] * len(mdl.residues)
        for (n, r) in enumerate(mdl.residues):
           try:
#              No need to divide by 2 here, since profile contributions are
#              already divided by the number of atoms (2 in the case of
#              2-atom statistical potentials)
               norm_prof[n] = prof.profile[n] - self.aa_scores[r.name]
               numstdres += 1
           except KeyError:
               pass
        correction = 2194.27 / float(numstdres)
        for n in range(len(mdl.residues)):
            norm_prof[n] -= correction
        return energy_profile(norm_prof, prof.nprofile, prof.min_rms,
                              prof.heavy_rms)

    def _get_stdev(self, numstdres):
        """Get the predicted DOPE score standard deviation"""
        return 14.3129 * numstdres

    def _get_mean(self, mdl):
        """Get the predicted DOPE score"""
        (score, numstdres) = self._get_aa_scores(mdl)
        return (score + self._get_small_chain_correction(numstdres), numstdres)

    def _get_aa_scores(self, mdl):
        """Get the contribution to the DOPE score from standard amino acids"""
        score = 0.
        numstdres = 0
        for r in mdl.residues:
            try:
                score += self.aa_scores[r.name]
                numstdres += 1
            except KeyError:
                pass
        return (score, numstdres)

    def _get_small_chain_correction(self, numstdres):
        """DOPE score is not linear for very short chains, so calculate a
           correcting term."""
        e = 2.7182818284590451  # we don't want to require the math module for e
        return 2236.45 - 2236.45 * e ** (-0.0432695 * numstdres)


class new_dope_scorer(dope_scorer):
    """Use raw DOPE scores to get normalized measures, using the new
       training set."""

    aa_scores = { 'ALA': -128.256,
                  'CYS': -94.8181,
                  'ASP': -23.3652,
                  'GLU': -63.8829,
                  'PHE': -158.393,
                  'GLY': -26.1659,
                  'HIS': -174.044,
                  'ILE': -194.958,
                  'LYS': -56.4597,
                  'LEU': -183.515,
                  'MET': -68.0257,
                  'ASN': -42.5151,
                  'PRO': -40.6333,
                  'GLN': -118.612,
                  'ARG': -42.9827,
                  'SER': 20.7157,
                  'THR': -100.901,
                  'VAL': -72.5706,
                  'TRP': -175.277,
                  'TYR': -366.699
                }

    def _get_stdev(self, numstdres):
        """Get the predicted DOPE score standard deviation"""
        return 14.12 * numstdres

    def _get_small_chain_correction(self, numstdres):
        """Add a constant term to the mean for the fit"""
        return 1625.87


class dopehr_scorer(dope_scorer):
    """Use raw DOPE-HR scores to get normalized measures, using the new
       training set."""

    aa_scores = { 'ALA': -160.524,
                  'CYS': -30.3241,
                  'ASP':  34.539,
                  'GLU': -45.4734,
                  'PHE': -131.283,
                  'GLY': 14.7563,
                  'HIS': -135.9,
                  'ILE': -160.324,
                  'LYS': -54.1812,
                  'LEU': -145.011,
                  'MET': -3.51822,
                  'ASN': 15.1741,
                  'PRO': 37.3899,
                  'GLN': -109.93,
                  'ARG': -35.056,
                  'SER': 9.42672,
                  'THR': -77.0285,
                  'VAL': -75.3469,
                  'TRP': -44.7595,
                  'TYR': -306.688
                }

    def _get_stdev(self, numstdres):
        """Get the predicted DOPE score standard deviation"""
        return 19.55 * numstdres

    def _get_small_chain_correction(self, numstdres):
        """Add a constant term to the mean for the fit"""
        return 322.112
