"""All physical restraint types, used for creating restraints, and in the
   schedule."""

class physical_type(object):
    """Physical restraint type"""
    _num = 0
    _types = []
    def __init__(self, doc):
        physical_type._num += 1
        self._num = physical_type._num
        self.__doc__ = doc
        physical_type._types.append(self)
    def get_num(self):
        return self._num
    def __str__(self):
        return '<physical restraint type: %s>' % self.__doc__
    def __repr__(self):
        return '<%s>' % self.__doc__
    modpt = property(get_num)


class values(object):
    """Scaling or cutoff values for all physical restraint types"""

    def __init__(self, default=1.0, **keys):
        self._default = default
        self._dict = {}
        for (term,val) in keys.items():
            term = eval("%s" % term)
            self[term] = val

    def set_from_list(self, lval):
        """Set the state from a Modeller internal list of values"""
        for (num, val) in enumerate(lval):
            typ = physical_type._types[num]
            self[typ] = val

    def __len__(self):
        return len(physical_type._types)

    def __repr__(self):
        if self._default is not None:
            terms = ["'default':%f" % self._default]
        else:
            terms = []
        for term in physical_type._types:
            num = term.get_num()
            if num in self._dict:
                terms.append("'%s':%f" % (term.__doc__, self._dict[num]))
        return "{" + ", ".join(terms) + "}"

    def __mul__(self, other):
        obj = values()
        if not isinstance(other, values):
            raise TypeError, "Must use physical.values objects here"
        obj._default = self._default * other._default
        for x in self._dict:
            if x in other._dict:
                oval = other._dict[x]
            else:
                oval = other._default
            obj._dict[x] = self._dict[x] * oval
        for x in other._dict:
            if x not in self._dict:
                obj._dict[x] = self._default * other._dict[x]
        return obj

    def _typecheck(self, item):
        if not isinstance(item, physical_type):
            raise TypeError, "keys must be physical_type objects or 'default'"

    def __contains__(self, item):
        if isinstance(item, str) and item == 'default':
            return True
        self._typecheck(item)
        return item in physical_type._types

    def __getitem__(self, key):
        # Allow read-only access as a list, for the convenience of old code
        # which expects a vector of numbers
        if isinstance(key, int):
            return self[physical_type._types[key]]
        if isinstance(key, str) and key == 'default':
            return self._default
        self._typecheck(key)
        try:
            return self._dict[key.get_num()]
        except KeyError:
            return self._default

    def __setitem__(self, key, value):
        if isinstance(key, str) and key == 'default':
            self._default = value
        self._typecheck(key)
        self._dict[key.get_num()] = value

def from_list(lval):
    """Initialize an object from a Modeller internal list of values"""
    obj = values(default=None)
    obj.set_from_list(lval)
    return obj

bond = physical_type("Bond length potential")
angle = physical_type("Bond angle potential")
dihedral = physical_type("Stereochemical cosine torsion potential")
improper = physical_type("Stereochemical improper torsion potential")
soft_sphere = physical_type("Soft-sphere overlap restraints")
lennard_jones = physical_type("Lennard-Jones 6-12 potential")
coulomb = physical_type("Coulomb point-point electrostatic potential")
h_bond = physical_type("H-bonding potential")
ca_distance = physical_type("Distance restraints 1 (CA-CA)")
n_o_distance = physical_type("Distance restraints 2 (N-O)")
phi_dihedral = physical_type("Mainchain Phi dihedral restraints")
psi_dihedral = physical_type("Mainchain Psi dihedral restraints")
omega_dihedral = physical_type("Mainchain Omega dihedral restraints")
chi1_dihedral = physical_type("Sidechain Chi_1 dihedral restraints")
chi2_dihedral = physical_type("Sidechain Chi_2 dihedral restraints")
chi3_dihedral = physical_type("Sidechain Chi_3 dihedral restraints")
chi4_dihedral = physical_type("Sidechain Chi_4 dihedral restraints")
disulfide_distance = physical_type("Disulfide distance restraints")
disulfide_angle = physical_type("Disulfide angle restraints")
disulfide_dihedral = physical_type("Disulfide dihedral angle restraints")
lower_distance = physical_type("Lower bound distance restraints")
upper_distance = physical_type("Upper bound distance restraints")
sd_mn_distance = physical_type("Distance restraints 3 (SDCH-MNCH)")
chi5_dihedral = physical_type("Sidechain Chi_5 dihedral restraints")
phi_psi_dihedral = physical_type("Phi/Psi pair of dihedral restraints")
sd_sd_distance = physical_type("Distance restraints 4 (SDCH-SDCH)")
xy_distance = physical_type("Distance restraints 5 (X-Y)")
nmr_distance = physical_type("NMR distance restraints 6 (X-Y)")
nmr_distance2 = physical_type("NMR distance restraints 7 (X-Y)")
min_distance = physical_type("Minimal distance restraints")
nonbond_spline = physical_type("Non-bonded spline restraints")
accessibility = physical_type("Atomic accessibility restraints")
density = physical_type("Atomic density restraints")
absposition = physical_type("Absolute position restraints")
dihedral_diff = physical_type("Dihedral angle difference restraints")
gbsa = physical_type("GBSA implicit solvent potential")
em_density = physical_type("EM density fitting potential")
saxs = physical_type("SAXS restraints")
symmetry = physical_type("Symmetry restraints")
