import _modeller
import util.top as top
import util.modutil as modutil
import residue
from modeller.util.modobject import modobject

class sequence_db(modobject):
    """Holds a database of protein sequences"""
    __modpt = None
    env = None
    top = None

    def __new__(cls, *args, **vars):
        obj = modobject.__new__(cls)
        obj.__modpt = _modeller.new_sequence_db(obj)
        return obj

    def __init__(self, env, **vars):
        self.env = env.copy()
        self.top = top.top(self.env)
        if len(vars) > 0:
            self.read(**vars)

    def __del__(self):
        _modeller.free_sequence_db(self.modpt)

    def __repr__(self):
        if len(self) == 1:
            return "Database of 1 sequence"
        else:
            return "Database of %d sequences" % len(self)

    def __str__(self):
        return "<%s>" % repr(self)

    def __get_modpt(self):
        return self.__modpt

    def __len__(self):
        return _modeller.sequence_db_nchn_get(self.modpt)

    def read(self, chains_list, seq_database_file, seq_database_format,
             clean_sequences=True, minmax_db_seq_len=(0, 999999)):
        """Reads in a database from a file"""
        return _modeller.read_sequence_db(self.modpt, self.env.libs.modpt,
                                          chains_list, seq_database_file,
                                          seq_database_format, clean_sequences,
                                          minmax_db_seq_len)

    def convert(self, chains_list, seq_database_file, seq_database_format,
                outfile, clean_sequences=True, minmax_db_seq_len=(0, 999999)):
        """Converts a database to binary format"""
        return _modeller.convert_sequence_db(self.modpt,
                                             self.env.libs.modpt, chains_list,
                                             seq_database_file,
                                             seq_database_format, outfile,
                                             clean_sequences, minmax_db_seq_len)

    def write(self, chains_list, seq_database_file, seq_database_format):
        """Writes out a database to a file"""
        return _modeller.write_sequence_db(self.modpt, self.env.libs.modpt,
                                           chains_list, seq_database_file,
                                           seq_database_format)

    def search(self, aln, io=None, **vars):
        """Search for similar sequences"""
        if io is None:
            io = self.env.io
        return self.top.sequence_search('sequence_db.search', sdb=self.modpt,
                                        aln=aln.modpt, io=io.modpt,
                                        libs=self.env.libs.modpt, **vars)

    def filter(self, **vars):
        """Cluster sequences by sequence-identity"""
        return self.top.seqfilter('sequence_db.filter', sdb=self.modpt,
                                  libs=self.env.libs.modpt, **vars)

    def __getitem__(self, indx):
        ret = modutil.handle_seq_indx(self, indx)
        if type(ret) is int:
            return sequence(self, indx)
        else:
            return [self[ind] for ind in ret]

    modpt = property(__get_modpt)


class sequence(object):
    """A single sequence in the database"""

    def __init__(self, sdb, num):
        self.sdb = sdb
        self.env = sdb.env
        self.num = num

    def __len__(self):
        nseq = _modeller.sequence_db_nseq_get(self.sdb.modpt)
        return _modeller.f_int1_get(nseq, self.num)

    def __repr__(self):
        if len(self) == 1:
            return "Sequence of 1 residue"
        else:
            return "Sequence of %d residues" % len(self)
    def __str__(self):
        return "<%s>" % repr(self)

    def __get_code(self):
        return _modeller.sequence_db_code_get(self.sdb.modpt, self.num)
    def __set_code(self, val):
        _modeller.sequence_db_code_set(self.sdb.modpt, self.num, val)
    def __get_prottyp(self):
        return _modeller.sequence_db_prottyp_get(self.sdb.modpt, self.num)
    def __set_prottyp(self, val):
        _modeller.sequence_db_prottyp_set(self.sdb.modpt, self.num, val)
    def __get_resol(self):
        resoldb = _modeller.sequence_db_resoldb_get(self.sdb.modpt)
        return _modeller.f_float1_get(resoldb, self.num)
    def __set_resol(self, val):
        resoldb = _modeller.sequence_db_resoldb_get(self.sdb.modpt)
        _modeller.f_float1_set(resoldb, self.num, val)
    def __get_residues(self):
        return residuelist(self)

    code = property(__get_code, __set_code, doc="Alignment code")
    prottyp = property(__get_prottyp, __set_prottyp, doc="Protein type")
    resol = property(__get_resol, __set_resol, doc="Resolution")
    residues = property(__get_residues)


class residuelist(object):

    def __init__(self, seq, offset=0, length=None):
        self.seq = seq
        self.offset = offset
        self.length = length

    def __len__(self):
        if self.length is not None:
            return self.length
        else:
            return len(self.seq)

    def __getitem__(self, indx):
        ret = modutil.handle_seq_indx(self, indx)
        if type(ret) is int:
            return seqdb_residue(self.seq, ret + self.offset)
        else:
            return [self[ind] for ind in ret]


class seqdb_residue(residue.residue):
    def __get_type(self):
        return _modeller.sequence_db_restype_get(self.mdl.sdb.modpt,
                                                 self.mdl.num, self._num)
    def __set_type(self, val):
        _modeller.sequence_db_restype_set(self.mdl.sdb.modpt,
                                          self.mdl.num, self._num, val)
    type = property(__get_type, __set_type, doc="Integer residue type")
