/** \file mod_file.h       File-handling routines.
 *
 *             Part of MODELLER, Copyright(c) 1989-2007 Andrej Sali
 */

#ifndef __MOD_FILE_H
#define __MOD_FILE_H

#include <stdio.h>
#include "mod_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/** Modeller-specific file attributes */
struct mod_file {
  /** Filename */
  const char *filename;
  /** Is the file compressed with gzip/bzip2 etc. (not HDF5 compress filters) */
  int compressed;
  /** File mode, i.e. "READ", "WRITE", "READWRITE" */
  char *action;
};

/** Open a file, and uncompress it if necessary. */
FILE *open_file(const char *path, const char *mode, struct mod_file *file_info);

/** Close an open file, and do any other necessary tidy-up if it was
    compressed. The initial value of ierr is used (if an error was already
    set, it is not modified, but emergency cleanup is done here). */
void close_file(FILE *fp, struct mod_file *file_info, int *ierr);

/** Expand out all environment variables in a filename. Additionally, the
    Modeller-specific ${LIB} and ${JOB} variables are expanded. The
    expanded string is returned, and should be freed when no longer needed. */
char *expand_filename(const char *filename);

/** Delete a file. */
void delete_file(const char *file, int *ierr);

/** Return TRUE iff the given file exists. */
mbool filexs(const char *file);

/** Deprecated interface to filexs() for TOP interpreter. */
void inquire(const char *file, int *file_exists);

/** Return all necessary information to deal with compressed files. */
void zfnam(const char *infil, const char *action, char **fname, char **zfname,
           char **unzipcmd, char **zipcmd, char **cpunzipcmd);

/** Uncompress the file, if already compressed */
char *prepfil(const char *infil, const char *action, mbool *compressed,
              int *ierr);

/** Puts the file in the original state. */
void unprepfil(const char *fname, const char *action, mbool compressed,
               int *ierr);

/** Return TRUE iff the given file, or a compressed version, exists */
mbool filexs_z(const char *filename);

/** Look for the given file, trying the directories and extensions given.
    The full path of the file is returned, if found, otherwise NULL. */
char *filexs_z_path(const char *filename, const char *dirs, const char *exts);

/** Interface to filexs_z_path for old TOP code */
char *fullfn(const char *filename, const char *dirs, const char *exts,
             int *ierr);

#ifdef __cplusplus
}
#endif
#endif  /* __MOD_FILE_H */
