/** \file mod_file.h       File-handling routines.
 *
 *             Part of MODELLER, Copyright(c) 1989-2007 Andrej Sali
 */

#ifndef __MOD_FILE_H
#define __MOD_FILE_H

#include <stdio.h>
#include "mod_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/** Modeller-specific file attributes */
struct mod_file {
  /** Requested filename */
  const char *filename;
  /** Is the file compressed with gzip/bzip2 etc. (not HDF5 compress filters) */
  gboolean compressed;
  /** File mode, i.e. "READ", "WRITE", "READWRITE" */
  char *action;
  /** Real filename after uncompression and expansion */
  char *realname;
};

/** Open a file, and uncompress it if necessary. */
FILE *mod_file_open(const char *path, const char *mode,
                    struct mod_file *file_info);

/** Close an open file, and do any other necessary tidy-up if it was
    compressed. The initial value of ierr is used (if an error was already
    set, it is not modified, but emergency cleanup is done here). */
void mod_file_close(FILE *fp, struct mod_file *file_info, int *ierr);

/** Read a single line from a file */
void mod_file_read_line(FILE *fp, GString *str, int *ierr, int *eof);

/** Read all data from a file */
void mod_file_read_contents(FILE *fp, char **text, unsigned *filelen,
                            int *ierr);

/** Expand out all environment variables in a filename. Additionally, the
    Modeller-specific ${LIB} and ${JOB} variables are expanded. The
    expanded string is returned, and should be freed when no longer needed. */
char *mod_file_expand(const char *filename);

/** Delete a file. */
void mod_file_delete(const char *file, int *ierr);

/** Return TRUE iff the given file exists. */
gboolean mod_file_exists(const char *file);

/** Deprecated interface to mod_file_exists() for TOP interpreter. */
void mod_inquire(const char *file, int *file_exists);

/** Uncompress the file, if already compressed */
char *mod_file_prep(const char *infil, const char *action,
                    gboolean *compressed, int *ierr);

/** Puts the file in the original state. */
void mod_file_unprep(const char *fname, const char *realname,
                     const char *action, gboolean compressed, int *ierr);

/** Return TRUE iff the given file, or a compressed version, exists */
gboolean mod_file_exists_z(const char *filename);

/** Look for the given file, trying the directories and extensions given.
    The full path of the file is returned, if found, otherwise NULL. */
char *mod_file_exists_z_path(const char *filename, const char *dirs,
                             const char *exts);

/** Deprecated interface to mod_file_exists_z_path for TOP interpreter */
char *mod_fullfn(const char *filename, const char *dirs, const char *exts,
                 int *ierr);

#ifdef __cplusplus
}
#endif
#endif  /* __MOD_FILE_H */
