/** \file mod_mdt_type.h   MDT types.
 *
 *             Part of MODELLER, Copyright(c) 1989-2007 Andrej Sali
 */

#ifndef __MOD_MDTTYPE_H
#define __MOD_MDTTYPE_H

/** MDT bin type */
struct mod_mdt_bin {
  /** Start and end of bin range */
  float rang1, rang2;
  /** Human-readable symbol */
  char *symbol;
};

/** MDT library feature */
struct mod_mdt_libfeature {
  /** Indices of data files/actions to be done for each feature in precalc() */
  int *idatfil;
  /** Proteins to read data from (1=protein 1, 2=protein 2, 3=proteins 1 and 2,
      4=protein 3, 5=proteins 1 and 3) */
  int iknown;
  /** Type of scan (0=whole protein, 1=single residue, 2=residue pair,
      3=atom per residue, 4=atom per residue pair, 5=single atom,
      6=atom pair, 7=atom triplet, 8=atom triplet pair) */
  int iresfeat;
  /** Type of precalculation required */
  int idatfeat;
  /** >1 if asymmetric (then N^2 loop, not (N*(N-1)/2)) */
  int isymm;
  /** The number of actions in precalc */
  int ndatfil;
  /** Feature name */
  char *name;
  /** The type of the feature in mdt.bin:
      0 ... no symbols, no ranges specified (hardwired);
      1 ... no ranges specified, only symbols;
      2 ... real ranges and symbols specified explicitly;
      3 ... real ranges specified with beginning and width (no symbols) */
  int itsymb;
  /** Number of bins (including undefined) */
  int nbins;
  /** Bin data; dimension(nbins) */
  struct mod_mdt_bin *bins;
};

/** MDT feature */
struct mod_mdt_feature {
  /** Starting bin -1 for each selected feature */
  int istart;
  /** Ending bin -1 for each selected feature */
  int iend;
  /** Number of used bins for each feature */
  int nbins;
  /** Integer type of each feature */
  int ifeat;
  /** Number of positions to skip for lookup of bin indices */
  int stride;
};

struct mod_mdt;
struct mod_mdt_library;

/** Data file types needed to be read in before evaluating a feature */
typedef enum {
  MOD_MDTF_STRUCTURE = 1,
  MOD_MDTF_SECSTRUC,
  MOD_MDTF_PSA,
  MOD_MDTF_NEIGHBORS,
  MOD_MDTF_CONTACT,
  MOD_MDTF_PHI,
  MOD_MDTF_PSI,
  MOD_MDTF_OMEGA,
  MOD_MDTF_CHI1,
  MOD_MDTF_CHI2,
  MOD_MDTF_CHI3,
  MOD_MDTF_CHI4,
  MOD_MDTF_DIHEDRALS,
  MOD_MDTF_CHI5
} mod_mdt_file;

/** Property to be calculated before evaluating a feature */
typedef enum {
  MOD_MDTC_CHI1 = 1,
  MOD_MDTC_CHI2,
  MOD_MDTC_PHI,
  MOD_MDTC_PSI,
  MOD_MDTC_MNCHCONF,
  MOD_MDTC_ACCESS,    /* single accessibility */
  MOD_MDTC_ID,        /* overall identity */
  MOD_MDTC_RESOL,     /* X-ray resolution */
  MOD_MDTC_NONE,
  MOD_MDTC_SCBISO,
  MOD_MDTC_HETCNT,
  MOD_MDTC_MNRAMA,    /* mainchain conformation (Ramachandran) */
  MOD_MDTC_SDCHCONF,
  MOD_MDTC_ALPHACONT, /* Alpha content class */
  MOD_MDTC_CHI3,
  MOD_MDTC_CHI4,
  MOD_MDTC_ATIND,     /* Atom indices for distance calculation */
  MOD_MDTC_CHI1CL,    /* chi1 class */
  MOD_MDTC_CHI2CL,
  MOD_MDTC_CHI3CL,
  MOD_MDTC_CHI4CL,
  MOD_MDTC_OMEGA,
  MOD_MDTC_PHICL,
  MOD_MDTC_PSICL,
  MOD_MDTC_OMEGACL,
  MOD_MDTC_CHI5,
  MOD_MDTC_CHI5CL,
  MOD_MDTC_DIST2,
  MOD_MDTC_ALPHA
} mod_mdt_calc;

/** Type of scan over available data for a feature */
typedef enum {
  MOD_MDTS_PROTEIN = 0,
  MOD_MDTS_RESIDUE,
  MOD_MDTS_RESIDUE_PAIR,
  MOD_MDTS_SPEC_ATOM,      /* one atom per residue */
  MOD_MDTS_SPEC_ATOM_PAIR,
  MOD_MDTS_ATOM,           /* any atom (first entry in alignment only) */
  MOD_MDTS_ATOM_PAIR,
  MOD_MDTS_TUPLE,          /* atom tuple (first entry in alignment only) */
  MOD_MDTS_TUPLE_PAIR,
  MOD_MDTS_BOND,
  MOD_MDTS_ANGLE,
  MOD_MDTS_DIHEDRAL
} mod_mdt_scan;

/** Protein(s) on which a given feature acts */
typedef enum {
  MOD_MDTP_A = 1,
  MOD_MDTP_B,
  MOD_MDTP_AB,
  MOD_MDTP_C,
  MOD_MDTP_AC
} mod_mdt_protein;

/** Register a new MDT library feature */
void mod_mdt_libfeature_register(struct mod_mdt_library *mlib, int ifeat,
                                 const char *name, mod_mdt_calc precalc_type,
                                 mod_mdt_protein protein_type,
                                 mod_mdt_scan scan_type, gboolean asymmetric,
                                 ...);

/** Redimension a single MDT library feature. */
void mod_mdt_libfeature_nbins_set(struct mod_mdt_libfeature *feat, int nbins);

/** Set the number of bin elements in an MDT. */
void mod_mdt_nelems_set(struct mod_mdt *mdt, int nelems);

/** Set the number of features in an MDT. */
void mod_mdt_nfeat_set(struct mod_mdt *mdt, int nfeat);

#endif   /* __MOD_MDTTYPE_H */
