# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

package PLLib::MD5Utils;
require Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw( GetMD5Seq AliMD5 ModMD5
            );
             

use strict;
use Digest::MD5;

use PLLib::Utils;

sub ModMD5 {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my $modfile = $_[0];

   # --- Check if the file exists
   unless ( -e $modfile ){
      warn "${subname}__E> File not found: $modfile\n";
      return;
   }

   # --- Get the model
   my $fh_mod = OpenFile( $modfile );
   my $model = undef;
   while ( my $line = <$fh_mod> ){
      # Ignore EXPDTA line, since it contains the date the model was
      # produced (and so changes every time); also ignore anything that
      # contains the ModPipe version number, as that is also subject to change
      if ($line !~ /^EXPDTA    THEORETICAL MODEL, MODELLER SVN/ &&
          $line !~ /^REMARK   6 GENERATED BY MODPIPE VERSION/) {
        $model .= $line;
      }
   }
   close( $fh_mod);

   # --- generate model digest
   my $md5= Digest::MD5->new;
   $md5->reset();
   $md5->add($model);
   my $md5id = $md5->hexdigest();

   return ($md5id);
}


sub AliMD5 {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my $alifile = $_[0];

   # --- Check if the file exists
   unless ( -e $alifile ){
      warn "${subname}__E> File not found: $alifile\n";
      return;
   }

   # --- Get the alignment
   my $fh_ali = OpenFile( $alifile );
   my $alignment = undef;
   while ( my $line = <$fh_ali> ){
      $alignment .= $line;
   }
   close( $fh_ali );

   # --- generate alignment digest
   my $md5= Digest::MD5->new;
   $md5->reset();
   $md5->add($alignment);
   my $md5id = $md5->hexdigest();

   return ($md5id);
}


sub GetMD5Seq {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my $sequence = $_[0];

   my $md5= Digest::MD5->new;

   # --- generate sequence digest
   $md5->reset();
   $md5->add($sequence);
   my $md5id = $md5->hexdigest();
   $md5id = $md5id . substr($sequence,0,4) . substr($sequence,-4);

   return ($md5id);
}

