#!/usr/bin/python
# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function
from modeller import *
from optparse import OptionParser
from shutil import rmtree
import modpipe.version
import modpipe.pdbutils
import sys, os, subprocess, tempfile, re

def calculate_no(pdbrep, natcod, natchn, modcod, modchn, outalif):
    """Calculates native overlap between a model and
    corresponding native structure."""

    # Search for PDB in specified path
    pdbf1 = modpipe.pdbutils.locate_PDB(natcod, pdbrep)
    pdbf2 = modpipe.pdbutils.locate_PDB(modcod, pdbrep)

    # -- Initialize some modeller stuff
    log.minimal()
    env = environ()
    env.io.atom_files_directory = pdbrep

    # Fetch the PDB chain objects for the two structures
    nat = modpipe.pdbutils.fetch_PDB_chain(env, natcod, natchn)
    mod = modpipe.pdbutils.fetch_PDB_chain(env, modcod, modchn)

    # Create an empty model & alignment objects
    mdl = model(env)
    aln = alignment(env)

    # Read in the native structure and add it to the alignment
    mdl.read(file=natcod,
             model_segment=('FIRST:'+nat.name, 'LAST:'+nat.name))
    aln.append_model(mdl, align_codes=natcod+nat.name,
                     atom_files=natcod)

    # Read in the modeled structure
    mdl.read(file=modcod,
             model_segment=('FIRST:'+mod.name, 'LAST:'+mod.name))
    aln.append_model(mdl, align_codes=modcod+nat.name,
                     atom_files=modcod)

    # Align the two sequences
    aln.align(gap_penalties_1d=(-3000,-1000), matrix_offset=0,
              local_alignment=True, rr_file='$(LIB)/id.sim.mat')

    # Write out alignment if requested
    if outalif:
        aln.write(file=outalif, alignment_format='pir')

    # Create the superposition
    mdl1 = model(env, file=natcod,
                 model_segment=aln[0].range)
    mdl2 = model(env, file=modcod,
                 model_segment=aln[1].range)
    sel = selection(mdl1).only_atom_types('CA')

    # Print a header
    columns = ['model', 'len', 'alnlen', 'native', 'len',
               'alnlen', 'mod_grmsd', 'mod_geqvp',
               '(mod_cutoff, mod_cutoff_rmsd, mod_cutoff_eqvp)*']

    # Format and print results
    results = "> %-5s %5d %5d %-5s %5d %5d " % (modcod,
                                      len(mod.residues),
                                      len(mdl2.residues),
                                      natcod+nat.name,
                                      len(nat.residues),
                                      len(mdl1.residues))

    # Now add the modeller numbers
    r = sel.superpose(mdl2, aln, rms_cutoff=3.5)
    results = results +  "%8.4f %6d %5.2f %8.4f %6d " % (r.rms,
              r.num_equiv_pos, 3.5, r.cutoff_rms, r.num_equiv_cutoff_pos)

    cuts = [1.0, 2.0, 3.0, 4.0, 5.0, 8.0]
    for c in cuts:
        r = sel.superpose(mdl2, aln, rms_cutoff=c)
        results = results +  "%5.2f %8.4f %6d " % ( c, r.cutoff_rms,
                                                   r.num_equiv_cutoff_pos)

    print('# ' + ' '.join(columns))
    return results


def main():

    # Parse command line options
    parser = OptionParser(version=modpipe.version.message())

    # Set defaults
    parser.set_usage("""
 This script takes two PDB protein chains and aligns them using CE. The
 resulting alignment is then also cleaned up to make it Modeller-friendly.
 The various structural overlap numbers are also reported.

 Usage: %prog [options]

 Run `%prog -h` for help information
 """)

    parser.set_defaults(native='',
                        natchn='',
                        model='',
                        modchn='',
                        outalif='',
                        )

    # Populate options list
    parser.add_option("-p", "--native",
                 dest="native",
                 type='string',
                 help="""PDB code of the native structure.
                      This is a mandatory option.""",
                 metavar="PDB")
    parser.add_option("-c", "--natchn",
                 dest="natchn",
                 type='string',
                 help="""PDB chain identifier of the native structure.
                      If not specified it will take the first chain.
                      """,
                 metavar="CHN")
    parser.add_option("-q", "--model",
                 dest="model",
                 type='string',
                 help="""PDB code of the modeled structure.
                      This is a mandatory option.""",
                 metavar="PDB")
    parser.add_option("-d", "--modchn",
                 dest="modchn",
                 type='string',
                 help="""PDB chain identifier of the modeled structure.
                      If not specified it will take the first chain.
                      """,
                 metavar="CHN")
    pdb = modpipe.pdbutils.get_pdb_repository(include_local=True)
    parser.add_option("-x", "--pdb_repository",
                      dest="pdbrep", type='string',
                      help="""Directory containing PDB files. The default
                              value is """ + pdb, default=pdb,
                      metavar="DIR")
    parser.add_option("-s", "--output_alignment_file",
                 dest="outalif",
                 type='string',
                 help="""File to store the alignment output.""",
                 metavar="FILE")

    # Check mandatory options
    opts, args = parser.parse_args()

    if not opts.native or not opts.model:
        parser.print_help()
        sys.exit(1)

    r = calculate_no(opts.pdbrep, opts.native, opts.natchn,
                 opts.model, opts.modchn, opts.outalif)
    print(r)


if __name__ == "__main__":
    main()
