# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

package PLLib::ModPy;
require Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw( WRTPY_RWSEQ WRTPY_SUPERPOSEAV WRTPY_SUPERPOSE 
              WRTPY_PPSCAN WRTPY_SUPERPOSEAV_NOALI
              WRTPY_WRITEDATA WRTPY_ALIGN WRTPY_ALIGN3D WRTPY_MODEL
              WRTPY_IDTABLE WRTPY_CHECKALIGNMENT
              WRTPY_SELFORLIGANDS WRTPY_GETSEQFROMPDB WRTPY_GETPDBXREF
              WRTPY_MODEL_WITH_SPBBRESTR);

use strict;
use PLLib::Utils;
use MPLib::Version;

# Return remark(s) for output PDB files containing information on the ModPipe
# version used to build them.
sub GetModelRemark {
  my $version = GetVersion();
  return "REMARK   6 GENERATED BY MODPIPE VERSION $version";
}

sub WRTPY_MODEL_WITH_SPBBRESTR {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 12;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $pyfh, $alnfile, $knowns, $tgtcode,
        $pdbrep, $begmod, $endmod, $assmethod,
        $seqid, $hetatm, $water, $hetres ) = @_;

   # --- Create the knowns string
   my $knowns_string = "";
   my $blank14 = "                        ";
   my $count = 0; my $template;
   foreach $template ( @$knowns ){
      $count++;
      $knowns_string .= "'" . $template . "'";
      $knowns_string .= ", " if ($count <= scalar(@$knowns));
      $knowns_string .= "\n" . $blank14 if ($count/4 == int($count/4));
   }

   # -- Pythonify some of the inputs
   $hetatm = ( $hetatm =~ /^ON$/i || $hetatm =~ /^TRUE$/i ) ? 'True' : 'False';
   $water = ( $water =~ /^ON$/i || $water =~ /^TRUE$/i ) ? 'True' : 'False';

   # -- Create special backbone restraints for each het residue
   my $rsrstr = '';
   if ( @$hetres > 0 ){
     $rsrstr = qq{
           def special_restraints(self, aln):
              rsr = self.restraints
              at = self.chains[0].atoms

                 };
              
     foreach my $r ( @$hetres ){
        my $n = $r + 1;
        $rsrstr .= qq{
              rsr.add(forms.gaussian(group=physical.bond,
                      feature=features.distance(at['C:$r'], at['N:$n']),
                      mean=1.3450, stdev=0.002))
              rsr.add(forms.gaussian(group=physical.angle,
                      feature=features.angle(at['O:$r'], at['C:$r'], at['N:$n']),
                      mean=2.0900, stdev=0.02))
              rsr.add(forms.gaussian(group=physical.dihedral,
                      feature=features.dihedral(at['CA:$r'], at['C:$r'], at['N:$n'], at['CA:$n']),
                      mean=3.1400, stdev=0.02))

                     };
     }
   }

   # --- Create the python script file with commands
   #     Redefine the summary writing routine. If things change in MODELLER
   #     this will have to get changed.
   my $model_remark = GetModelRemark();
   my $pystr = undef;
   $pystr =    qq{
         from __future__ import print_function
         from modeller import *
         from modeller.automodel import *
         log.minimal()
         env = environ()
         env.io.atom_files_directory = '$pdbrep'
         env.io.hetatm = $hetatm
         env.io.water = $water

         def custom_randomize(atmsel):
             """Randomize all coordinates"""
             mdl = atmsel.get_model()
             oursel = atmsel - atmsel.only_het_residues()
             oursel.randomize_xyz(deviation=mdl.deviation)

         class MyAutoModel(automodel):
           def user_after_single_model(self):
               self.remark = """$model_remark"""

           def write_ok_summary(self, all, modeltyp):
               """Print out a detailed summary of all successfully 
                  generated models"""
               print()
               print(">> Summary of successfully produced %s:" % modeltyp)
               columns = ['molpdf', 'DOPE score', 'DOPE-HR score',
                          'GA341', 'Compactness', 'energy_pair',
                          'energy_surf', 'energy_comb', 'z_pair',
                          'z_surf', 'z_comb', 'norm_DOPE']
               header = ('     %-55s' % 'Filename'
	                 + ''.join(' %14s' % a for a in columns))
               print(header)
               print('-' * len(header))

               fields = ['molpdf', 'DOPE score', 'DOPE-HR score', 
                         'GA341 score', 'Normalized DOPE score']
               for mdl in all:
                   text = 'SCR> %-55s' % mdl['name']
                   for field in fields:
                       if field == 'GA341 score':
                           for score in mdl[field]:
                               text = text + ' %14.5f' % score
                       else:
                           text = text + ' %14.5f' % mdl[field]
                   print(text)
               print(">> END OF SUMMARY")

                 };

   
   # -- Paste the restraints stuff
   $pystr .= $rsrstr;

   $pystr .=   qq{
         a = MyAutoModel(env,
                         alnfile = '$alnfile',
                         knowns = ( $knowns_string ),
                         sequence = '$tgtcode')

         a.starting_model = $begmod
         a.ending_model   = $endmod

         a.assess_methods = (assess.GA341, 
                             assess.DOPE,
                             assess.DOPEHR,
                             assess.normalized_dope)

         a.rand_method = custom_randomize
         a.make()
                 };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return
   return 1;
}


sub WRTPY_GETPDBXREF {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 10;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($pyfh, $pdbrep, $inppdb, $range_beg, $range_end,
       $hetatm, $water) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.minimal()
         env = environ()
         env.io.atom_files_directory = '$pdbrep'
         env.io.hetatm = $hetatm
         env.io.water = $water

         mdl = model(env, file='$inppdb', model_segment=('$range_beg', '$range_end'))
         for r in mdl.residues:
           print "IDX> %6d %6s:%s  %s  %s" % (r.index, r.num, 
                                              r.chain.name, r.pdb_name, 
                                              r.code)
                  };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return value
   return 1;
}



sub WRTPY_GETSEQFROMPDB {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 10;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($pyfh, $pdbrep, $inppdb, $range_beg, $range_end,
       $acode, $afile, $outfile, $hetatm, $water) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.minimal()
         env = environ()
         env.io.atom_files_directory = '$pdbrep'
         env.io.hetatm = $hetatm
         env.io.water = $water

         aln = alignment(env)
         mdl = model(env, file='$inppdb', model_segment=('$range_beg', '$range_end'))

         aln.append_model(mdl, align_codes='$acode', atom_files='$afile')
         aln.write(file='$outfile')
                  };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return value
   return 1;
}


sub WRTPY_SELFORLIGANDS {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 11;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($pyfh, $pdbrep, $inppdb, $outdir, $outpdb,
       $alibeg, $aliend, $spherecut, $watercut, 
       $hetatm, $water) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         import re
         log.minimal()
         env = environ()
         env.io.atom_files_directory = '$pdbrep'
         env.io.hetatm = $hetatm
         env.io.water = $water
         
         m = model(env, file='$inppdb')
         # Handle +xx alignment notation
         mtch = re.match('\\s*\\+(\\d+)\\s*:', '$aliend')
         if mtch:
             r = m.residues['$alibeg']
             rr = m.residue_range(r._num, r._num + int(mtch.group(1)) - 1)
             assert(rr[0]._num == r._num)
             s = selection(rr)
         else:
             s = selection(m.residue_range('$alibeg', '$aliend'))
         
         def select_hetatms_in_protein_range(sel, cutoff):
            """Selects all hetatms within a given range of the protein"""
            s = sel.select_sphere(cutoff)
            t = s.only_het_residues()
            s = t.by_residue()
            return s
         
         def filter_selection_for_ligands(sel):
            """Filters a given selection to pick only ligand atoms"""
            s = sel - sel.only_residue_types('TIP3')
            return s
         
         def filter_selection_for_waters(sel):
            """Filters a given selection to pick only water atoms"""
            s = sel.only_residue_types('TIP3')
            return s
         
         def select_waters_in_protein_range(sel, cutoff):
            """Selects all waters within a given range of the protein"""
            s = sel.select_sphere(cutoff)
            t = s.only_residue_types('TIP3')
            return t
         
         def fetch_ligand_water_contacts(lig, wat, cutoff):
            """Fetches all the waters within a given cutoff of the ligand"""
            s = lig.select_sphere(cutoff) & wat
            return s
         
         if env.io.hetatm is True and env.io.water is True:
            hets = select_hetatms_in_protein_range(s, $spherecut)
            ligs = filter_selection_for_ligands(hets)
            wats = select_waters_in_protein_range(s, $watercut)
            lwcs = fetch_ligand_water_contacts(ligs, wats, $watercut)
            if len(ligs) > 0:
               s.add(ligs)
            if len(lwcs) > 0:
               s.add(lwcs)
         elif env.io.hetatm is True and env.io.water is False:
            hets = select_hetatms_in_protein_range(s, $spherecut)
            ligs = filter_selection_for_ligands(hets)
            if len(ligs) > 0:
               s.add(ligs)
         elif env.io.hetatm is False and env.io.water is True:
            wats = select_waters_in_protein_range(s, $watercut)
            if len(wats) > 0:
               s.add(wats)
         
         s.write(file='$outdir/$outpdb')
                  };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return value
   return 1;
}



sub WRTPY_CHECKALIGNMENT {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 3;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($pyfh, $pdbrep, $alifile) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.minimal()
         env = environ()
         env.io.atom_files_directory = '$pdbrep'

         aln = alignment(env)
         aln.append(file='$alifile', align_codes='all')
         aln.check()
                  };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return value
   return 1;
}


sub WRTPY_IDTABLE {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 5;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($pyfh, $alifile, $aliformat, $matfile, $pdbrep) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.minimal()
         env = environ()
         env.io.atom_files_directory = ['$pdbrep']

         aln = alignment(env, file='$alifile')
         aln.id_table(matrix_file='$matfile')
                  };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return value
   return 1;
}


sub WRTPY_MODEL {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 10;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $pyfh, $alnfile, $knowns, $tgtcode,
        $pdbrep, $begmod, $endmod, $assmethod,
        $seqid, $hetatm ) = @_;

   # --- Create the knowns string
   my $knowns_string = "";
   my $blank14 = "                        ";
   my $count = 0; my $template;
   foreach $template ( @$knowns ){
      $count++;
      $knowns_string .= "'" . $template . "'";
      $knowns_string .= ", " if ($count <= scalar(@$knowns));
      $knowns_string .= "\n" . $blank14 if ($count/4 == int($count/4));
   }

   # -- Pythonify some of the inputs
   $hetatm = ( $hetatm =~ /^ON$/i || $hetatm =~ /^TRUE$/i ) ? 'True' : 'False';

   # --- Create the python script file with commands
   #     Redefine the summary writing routine. If things change in MODELLER
   #     this will have to get changed.
   my $model_remark = GetModelRemark();
   my $pystr = undef;
   $pystr =    qq{
         from __future__ import print_function
         from modeller import *
         from modeller.automodel import *
         log.minimal()
         env = environ()
         env.io.atom_files_directory = '$pdbrep'
         env.io.hetatm = $hetatm

         class MyAutoModel(automodel):
           def user_after_single_model(self):
               self.remark = """$model_remark"""
               aln = self.read_alignment()  
               seq = aln[-1] 
               r = seq.range
               (res, chain) = r[0].split(":") 
               self.rename_segments(segment_ids=chain, 
                                    renumber_residues=int(res))

           def write_ok_summary(self, all, modeltyp):
               """Print out a detailed summary of all successfully 
                  generated models"""
               print()
               print(">> Summary of successfully produced %s:" % modeltyp)
               columns = ['molpdf', 'DOPE score', 'DOPE-HR score',
                          'GA341', 'Compactness', 'energy_pair',
                          'energy_surf', 'energy_comb', 'z_pair',
                          'z_surf', 'z_comb', 'norm_DOPE']
               header = ('     %-55s' % 'Filename'
	                 + ''.join(' %14s' % a for a in columns))
               print(header)
               print('-' * len(header))

               fields = ['molpdf', 'DOPE score', 'DOPE-HR score', 
                         'GA341 score', 'Normalized DOPE score']
               for mdl in all:
                   text = 'SCR> %-55s' % mdl['name']
                   for field in fields:
                       if field == 'GA341 score':
                           for score in mdl[field]:
                               text = text + ' %14.5f' % score
                       else:
                           text = text + ' %14.5f' % mdl[field]
                   print(text)
               print(">> END OF SUMMARY")

         a = MyAutoModel(env,
                         alnfile = '$alnfile',
                         knowns = ( $knowns_string ),
                         sequence = '$tgtcode')

         a.starting_model = $begmod
         a.ending_model   = $endmod

         a.assess_methods = (assess.GA341, 
                             assess.DOPE,
                             assess.DOPEHR,
                             assess.normalized_dope)

         a.make()
                 };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return
   return 1;
}

sub WRTPY_ALIGN3D {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 7;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $pyfh, $seqsfile, $alncodes, $opgp, $exgp,
        $matfile, $alnfile ) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.minimal()
         env = environ()

         aln = alignment(env)
         aln.append(file='$seqsfile', 
                    align_codes=($alncodes))

         aln.align(rr_file='$matfile', 
                   gap_penalties_1d=($opgp,$exgp))
         aln.align3d(gap_penalties_3d=[0, 4.0])\n";
         aln.write(file='$alnfile')\n\n";

         name1 = aln[0].name
         name2 = aln[1].name
         pdb1 = aln[0].atom_file
         pdb2 = aln[1].atom_file

         mdl = model(env,file=pdb1, aln=aln, 
                     model_segment=(name1, name1))
         mdl.pick_atoms(aln, atom_types='CA')
         mdl2 = model(env, file=pdb2, aln=aln, 
                      model_segment=(name2, name2))
         mdl.superpose(mdl2, aln)
                 };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return
   return 1;
}

sub WRTPY_ALIGN {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 7;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $pyfh, $seqsfile, $alncodes, $opgp, $exgp,
        $matfile, $alnfile ) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.minimal()
         env = environ()

         aln = alignment(env)
         aln.append(file='$seqsfile', 
                    align_codes=($alncodes))
         aln.align(rr_file='$matfile', 
                   gap_penalties_1d=($opgp,$exgp))
         aln.write(file='$alnfile')
                  };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return
   return 1;
}

sub WRTPY_SUPERPOSEAV_NOALI {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 12;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($pyfh, 
       $acode1, $pdbf1, $beg1, $end1, $chn1,
       $acode2, $pdbf2, $beg2, $end2, $chn2,
       $outfile) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.minimal()
         env = environ()
                  };

   $pystr .=   qq{
         aln  = alignment(env)

         mdl  = model(env, file='$pdbf1', model_segment=('$beg1:$chn1', '$end1:$chn1'))
         aln.append_model(mdl, atom_files='$pdbf1', align_codes='$acode1')

         mdl2 = model(env, file='$pdbf2', model_segment=('$beg2:$chn2', '$end2:$chn2'))
         aln.append_model(mdl2, atom_files='$pdbf2', align_codes='$acode2')

         # -- Make atom selection
         atmsel = selection(mdl).only_atom_types('CA')

         # -- Do superposition
         atmsel.superpose(mdl2, aln, rms_cutoff=3.5)
         atmsel.superpose(mdl2, aln, rms_cutoff=1.0)
         atmsel.superpose(mdl2, aln, rms_cutoff=2.0)
         atmsel.superpose(mdl2, aln, rms_cutoff=3.0)
         atmsel.superpose(mdl2, aln, rms_cutoff=4.0)
         atmsel.superpose(mdl2, aln, rms_cutoff=5.0)
                    };

   # -- Write out the transformed coordinates
   $pystr .=   qq{
         mdl2.write(file='$outfile')
                 } if ( $outfile ne '');


   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return value
   return 1;
}

sub WRTPY_PPSCAN {

   use File::Basename;

   # --- Get subroutine name
   my $subname = GetSubrName();

   unless ( scalar(@_) >= 19){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($pyfh, $tgtprf, $prfformat,
       $pssmdbname, $pssmdbfmt, $prflist,
       $matoffset, $ccmatoffset, $rrfile, 
       $gap_i, $gap_e, $scrstat, $alncalc,
       $scrfile, $pssmwght, $evcut, $alnbase, 
       $summfile, $scoretype, $allow_statfail ) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.minimal()
         env = environ()
                  };

   # --- Convert the format of the input profile if
   #     necessary (like when reading a PSI-Blast generated
   #     profile
   
   if ( $prfformat =~ /\bFASTA\b/i || $prfformat =~ /\bPIR\b/i ){

      my $newprf = (fileparse($tgtprf, '\..*'))[0] . '.prf';
      my $format_convert = qq{
         aln = alignment(env)
         aln.append(file='$tgtprf', 
                    alignment_format='$prfformat', 
                    align_codes='ALL')

         prf = aln.to_profile()
         prf.write(file='$newprf',profile_format='TEXT')
                          };

      # -- Re-assing a couple of variables
      $prfformat = 'TEXT';
      $tgtprf = $newprf;

      # -- Add it to the python string
      $pystr .= $format_convert;
   } 

   my $errhandle;
   if ($allow_statfail) {
      $errhandle = "pass";
   } else {
      $errhandle = "raise";
   }
   $pystr .=   qq{
         prf = profile(env, file='$tgtprf', profile_format='$prfformat')
         psm = pssmdb(env, pssmdb_name='$pssmdbname', pssmdb_format='$pssmdbfmt')

         try:
             prf.scan(profile_list_file ='$prflist',
                  psm               = psm,
                  matrix_offset     = $matoffset,
                  ccmatrix_offset   = $ccmatoffset,
                  rr_file           = '\${LIB}/$rrfile',
                  gap_penalties_1d  = ($gap_i, $gap_e),
                  score_statistics  = $scrstat,
                  output_alignments = $alncalc,
                  output_score_file = '$scrfile',
                  profile_format    = '$prfformat',
                  max_aln_evalue    = $evcut,
                  aln_base_filename = '$alnbase',
                  pssm_weights_type = '$pssmwght',
                  summary_file      = '$summfile',
                  score_type        = '$scoretype')
         except StatisticsError as err:
             $errhandle
                    };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return value
   return 1;
}

sub WRTPY_RWSEQ {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 5;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($pyfh, $inpfile, $inpfmt, $outfile, $outfmt) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.minimal()
         env = environ()
                  };

   # -- Case 1: Input & Output formats are either fasta/pir
   if ( ($inpfmt =~ /^FASTA/i || $inpfmt =~ /^PIR/i ) &&
        ($outfmt =~ /^FASTA/i || $outfmt =~ /^PIR/i  )){
      $pystr .=   qq{
         sdb = sequence_db(env)
         sdb.read(seq_database_file='$inpfile',
                  seq_database_format='$inpfmt',         
                  chains_list='all',
                  minmax_db_seq_len=[1, 99999],
                  clean_sequences=False)
         sdb.write(seq_database_file='$outfile',
                   seq_database_format='$outfmt',
                   chains_list='all')
                    };
   }

   # -- Case 2: Input format is a profile
   if ( $inpfmt =~ /^PRF/i &&
        ($outfmt =~ /^FASTA/i || $outfmt =~ /^PIR/i  )){
      $pystr .=   qq{
         prf = profile(env)
         prf.read(file='$inpfile',
                  profile_format='text')
         aln = prf.to_alignment()
         aln.write(file='$outfile', alignment_format='$outfmt')
                    };
   }

   # -- Case 3: Output format is a profile
   if ( ($inpfmt =~ /^FASTA/i || $inpfmt =~ /^PIR/i ) &&
        $outfmt =~ /^PRF/i ){
      $pystr .=   qq{
         aln = alignment(env)
         aln.append(file='$inpfile', 
                    alignment_format='$inpfmt',
                    align_codes='all')
         prf = aln.to_profile(clean_sequences=False)
         prf.write(file='$outfile', profile_format='text')
                    };
   }

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return value
   return 1;
}

sub WRTPY_SUPERPOSEAV {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 10;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($pyfh, $alifile, $pdbf1, $beg1, $end1, $chn1,
       $pdbf2, $beg2, $end2, $chn2) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.minimal()
         env = environ()
                  };

   $pystr .=   qq{
         mdl  = model(env, file='$pdbf1',
                      model_segment=('$beg1:$chn1', '$end1:$chn1'))
         mdl2 = model(env, file='$pdbf2',
                      model_segment=('$beg2:$chn2', '$end2:$chn2'))
         aln  = alignment(env, file='$alifile', align_codes='all')

         # -- Make atom selection
         atmsel = selection(mdl).only_atom_types('CA')

         # -- Do superposition
         atmsel.superpose(mdl2, aln, rms_cutoff=1.0)
         atmsel.superpose(mdl2, aln, rms_cutoff=2.0)
         atmsel.superpose(mdl2, aln, rms_cutoff=3.0)
         atmsel.superpose(mdl2, aln, rms_cutoff=4.0)
         atmsel.superpose(mdl2, aln, rms_cutoff=5.0)
                    };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return value
   return 1;
}


sub WRTPY_SUPERPOSE {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 24;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($pyfh, $alifile, $pdbf1, $beg1, $end1, $chn1,
       $pdbf2, $beg2, $end2, $chn2, $fit, $refine, 
       $refine_local, $rmscut) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.minimal()
         env = environ()
                  };

   $pystr .=   qq{
         mdl  = model(env, file='$pdbf1',
                      model_segment=('$beg1:$chn1', '$end1:$chn1'))
         mdl2 = model(env, file='$pdbf2',
                      model_segment=('$beg2:$chn2', '$end2:$chn2'))
         aln  = alignment(env, file='$alifile', align_codes='all')

         # -- All C-alpha atoms
         atmsel = selection(mdl).only_atom_types('CA')
         atmsel.superpose(mdl2, aln, fit=$fit, superpose_refine=$refine, 
                          refine_local=$refine_local, rms_cutoff=$rmscut)

         # -- All backbone atoms
         atmsel = selection(mdl).only_mainchain()
         atmsel.superpose(mdl2, aln, fit=$fit, superpose_refine=$refine, 
                          refine_local=$refine_local, rms_cutoff=$rmscut)

         # -- All sidechain atoms
         atmsel = selection(mdl).only_sidechain()
         atmsel.superpose(mdl2, aln, fit=$fit, superpose_refine=$refine, 
                          refine_local=$refine_local, rms_cutoff=$rmscut)

         # -- All atoms
         atmsel = selection(mdl)
         atmsel.superpose(mdl2, aln, fit=$fit, superpose_refine=$refine, 
                          refine_local=$refine_local, rms_cutoff=$rmscut)
                    };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return value
   return 1;
}

sub WRTPY_WRITEDATA {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 5;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $pyfh, $pdbfile, $basename, $radii, $output ) = @_;

   # --- Create python file with commands
   my $pystr = undef;
   $pystr =    qq{
         from modeller import *
         log.verbose()
         env = environ()
         env.io.hetatm = False
         env.io.water = False
         env.libs.topology.read(file='\${LIB}/top_heav.lib')
                  };

   $pystr .=   qq{
         mdl = model(env, file='$pdbfile')

         myedat = energy_data()
         myedat.radii_factor = $radii
         mdl.write_data(file='$basename', 
                        edat=myedat, output='$output')
                  };

   # -- Reformat the string and write to file
   $pystr =~ s/^[^\S\n]{9}//gm;
   print $pyfh $pystr, "\n";
   
   # --- Return
   return 1;
}

