# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function
from modeller import *
from modeller.automodel import *
from modpipe.alnutils import parse_modeling_alignment
from modpipe.automodeloutput import *

def build_models(env, alnfile, number_of_models=1,
                 return_scores_for='best',
                 best_model_by='molpdf'):
    """This routine will take an alignment file and some
    specifications for the model building process and return
    the scores for all or the best models calculated."""
    (templates, target) = parse_modeling_alignment(env, alnfile)

    a = MyAutoModel(env, alnfile = alnfile, knowns = templates,
                    sequence = target)
    a.starting_model = 1
    a.ending_model   = a.starting_model + number_of_models - 1
    a.assess_methods = (assess.GA341, assess.DOPE, assess.DOPEHR,
                        assess.normalized_dope)
    a.make()

    p = automodeloutput(a.outputs)
    p.expand_GA341()
    p.trim_outputs()

    results = ""
    if return_scores_for == 'all':
        results = p.get_outputs()
    elif return_scores_for == 'best':
        p.select_best_model_by(best_model_by)
        results = p.get_outputs()

    return results


class MyAutoModel(automodel):
    def user_after_single_model(self):
        aln = self.read_alignment()
        seq = aln[-1]
        r = seq.range
        (res, chain) = r[0].split(":")
        self.rename_segments(segment_ids=chain,
                             renumber_residues=int(res))

    def write_ok_summary(self, all, modeltyp):
        """Print out a detailed summary of all successfully
           generated models"""
        print()
        print(">> Summary of successfully produced %s:" % modeltyp)
        columns = ['molpdf', 'DOPE score', 'DOPE-HR score',
                   'GA341', 'Compactness', 'energy_pair',
                   'energy_surf', 'energy_comb', 'z_pair',
                   'z_surf', 'z_comb', 'norm_DOPE']
        header = '     %-55s' % 'Filename' + \
                 " ".join(['%14s' % x for x in columns])
        print(header)
        print('-' * len(header))

        fields = ['molpdf', 'DOPE score', 'DOPE-HR score',
                  'GA341 score', 'Normalized DOPE score']
        for mdl in all:
            text = 'SCR> %-55s' % mdl['name']
            for field in fields:
                if field == 'GA341 score':
                    for score in mdl[field]:
                        text = text + ' %14.5f' % score
                else:
                    text = text + ' %14.5f' % mdl[field]
            print(text)
        print(">> END OF SUMMARY")
