#!/usr/bin/python
# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function
from modeller import *
from optparse import OptionParser
from tempfile import mkstemp
import modpipe.pdbutils
from modpipe.alnutils import *
from modpipe.pdbutils import *
from modpipe.suputils import *
from modpipe.scripts.align3d import calculate_a3d_alignment
import modpipe.version
import sys, os

def main():

    # Parse command line options
    parser = OptionParser(version=modpipe.version.message())

    # Set defaults
    parser.set_usage("""
 This is a batch version of align3d.py. It takes a query structure
 and a list of other structures and performs comparisons between them.

 Usage: %prog [options]

 Run `%prog -h` for help information
 """)

    # Populate options list
    parser.add_option("-p", "--pdb1", dest="pdb1", type='string',
                      help="""PDB code of the first structure.
                              This is a mandatory option.""",
                      metavar="PDB", default=None)
    parser.add_option("-c", "--chn1", dest="chn1", type='string',
                      help="""PDB chain identifier of the first structure.
                              If not specified it will take the first chain.""",
                      default='', metavar="CHN")
    parser.add_option("-l", "--list", dest="listf", type='string',
                      help="""File containing list of chains to scan
                              against. The list should be in the format
                              <pdbcode> <chain-id>, one pair per line.
                              This is a mandatory option.""",
                      metavar="FILE", default=None)
    pdb = modpipe.pdbutils.get_pdb_repository(include_local=True)
    parser.add_option("-x", "--pdb_repository", dest="pdbrep", type='string',
                      help="""Directory containing PDB files. The default
                              value is """ + pdb, default=pdb,
                      metavar="DIR")
    parser.add_option("-o", "--output_alignment", dest="outali",
                      action='store_true',
                      help="""Flag to store the final alignment.""",
                      default=False)
    parser.add_option("-z", "--output_results_file", dest="resf",
                      type='string',
                      help="""Filename for storing structure overlap results""",
                      metavar="FILE", default=None)

    # Check mandatory options
    opts, args = parser.parse_args()

    if not opts.pdb1 or not opts.listf:
        parser.print_help()
        sys.exit(1)

    if not os.path.isfile(opts.listf):
        parser.exit(1, 'List file '+opts.listf+' not found\n')

    return opts, args


if __name__ == "__main__":
    opts, args = main()
    # Open file stream
    if opts.resf:
        o = open(opts.resf, 'w')
    else:
        o = sys.stdout

    # Process the list
    f = open(opts.listf, 'r')
    for line in f:
        li = line.split()
        if len(li) < 2:
            li.append('')

        a3dalif=''
        if opts.outali:
            a3dalif = opts.pdb1 + opts.chn1 + '-' + li[0] + li[1] + '-align3d.ali'

        # Call function to calculate TMalign alignments and parse them
        try:
            header, results = calculate_a3d_alignment(opts.pdbrep, opts.pdb1,
                                opts.chn1, li[0], li[1], a3dalif, (0,0,0,1,0))
        except:
            print("Failed calculating alignment b/w %s and %s" % \
                    (opts.pdb1+opts.chn1, li[0]+li[1]))

        o.write(results+'\n')
    f.close()
