#!/usr/bin/python
# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

from modpipe.clustermodels import ClusterModels
from optparse import OptionParser
import modpipe.version
import sys

def main():

    psr = OptionParser(version=modpipe.version.message())

    psr.set_usage("""
  This is a wrapper script around the ClusterModels class. It takes
  specified command line options and clusters a list of models by
  comparison of structure overlaps.

  Usage: %prog [options]

  Run `%prog -h` for help information
  """)

    psr.set_defaults(listf='', mdir='', socut=60.0, clsfile='',
         repfile='', lrgfile='', cmpfile='', mineqv=30, maxnoneqv=30,
         rms_cutoff=3.5, sortmodels=True)

    psr.add_option('-i', dest='listf', type='string',
         help="""file with list of models""", metavar='FILE')
    psr.add_option('-d', dest='mdir', type='string',
         help="""path to directory containing models""", metavar='DIR')

    psr.add_option('-s', dest='sortmodels', action='store_true',
         help="""sort models by length before clustering""")
    psr.add_option('-c', '--so_cutoff', dest='socut', type='float',
         help="""cut-off structure overlap for clustering""", metavar='60.0')
    psr.add_option('-r', '--rms_cutoff', dest='rms_cutoff', type='float',
         help="""cut-off rms value to count two positions as equivalent""", metavar='3.50')
    psr.add_option('-m', '--min_eqv_pos', dest='mineqv', type='int',
         help="""min number of equivalent positions for clustering""", metavar='30')
    psr.add_option('-M', '--max_noneqv_pos', dest='maxnoneqv', type='int',
         help="""max number of non-equivalent positions for clustering""", metavar='30')

    psr.add_option('--clsfile', dest='clsfile', type='string',
         help="""filename to write clusters into""")
    psr.add_option('--repfile', dest='repfile', type='string',
         help="""filename to write representatives into""")
    psr.add_option('--lrgfile', dest='lrgfile', type='string',
         help="""filename to write the largest cluster into""")
    psr.add_option('--cmpfile', dest='cmpfile', type='string',
         help="""filename to write the details of the comparisons""")

    opts, args = psr.parse_args()

    if not opts.listf or not opts.mdir:
        psr.print_help()
        sys.exit(1)


    cls = ClusterModels(opts.listf, opts.mdir, sort_models=opts.sortmodels,
                        rms_cutoff=opts.rms_cutoff, min_equiv_pos=opts.mineqv,
                        max_nonequiv_pos=opts.maxnoneqv, so_cutoff=opts.socut)
    cls.make()

    cls.write_clusters(file=opts.clsfile)
    cls.write_representatives(file=opts.repfile)
    cls.write_largest_cluster(file=opts.lrgfile)
    cls.write_comparisons(file=opts.cmpfile)

if __name__ == "__main__":
    main()
