#!/usr/bin/python
# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function
from optparse import OptionParser
import modpipe.version
import modpipe.sequence
import os
import sys

sys.argv[0] = 'modpipe convert'

def get_options():
    parser = OptionParser(version=modpipe.version.message())

    parser.set_usage("""
 Converts between alignment file formats.

 Usage: %prog [options] infile

 infile is the input sequence file.

 Run `%prog -h` for help information
""")
    parsers = {'FASTA': modpipe.sequence.FASTAFile,
               'PIR': modpipe.sequence.PIRFile,
               'SPTR': modpipe.sequence.SPTRFile}
    in_parsers = {}
    out_parsers = {}
    for (key, value) in parsers.items():
        if hasattr(value, 'read'):
            in_parsers[key] = value
        if hasattr(value, 'write'):
            out_parsers[key] = value

    parser.add_option("-f", "--input_format", dest="infmt", metavar="FORMAT",
                      type="choice",
                      help="Format of the input file (%s, default PIR)" \
                           % ", ".join(in_parsers.keys()),
                      choices=list(in_parsers.keys()), default="PIR")
    parser.add_option("-g", "--output_format", dest="outfmt", metavar="FORMAT",
                      type="choice",
                      help="Format of the output file (%s, default FASTA)" \
                           % ", ".join(out_parsers.keys()),
                      choices=list(out_parsers.keys()), default="FASTA")
    parser.add_option("-o", "--output_file", dest="outfile", metavar="FILE",
                      type="string",
                      help="""Name of the output file; if unspecified,
                              print to standard out""",
                      default=None)
    opts, args = parser.parse_args()

    if len(args) != 1:
        parser.error("You must specify an input file name")
    infile = open(args[0], 'r')
    out_parser = out_parsers[opts.outfmt]()
    in_parser = in_parsers[opts.infmt]()
    if opts.outfile:
        outfile = open(opts.outfile, 'w')
    else:
        outfile = sys.stdout
    return infile, outfile, in_parser, out_parser


def main():
    infile, outfile, in_parser, out_parser = get_options()

    for (count, seq) in enumerate(in_parser.read(infile)):
        seq.clean()
        if count > 0 and count % 1000 == 0:
            print("ConvertSeq.py__M> Processed %d sequences" % count)
        out_parser.write(outfile, seq)


if __name__ == '__main__':
    main()
