# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.


import unittest
import sys
import os.path
import subprocess
import modpipe.binaries

class TestCase(unittest.TestCase):

    def run_script(self, dir, script, args):
        """Run a ModPipe script (a module in modpipe.dir), as if run from
           the command line, passing the given command line arguments.
           The script's stderr is redirected to stdout, so as not to clutter the
           test output."""
        required_prefix = 'modpipe.%s.' % dir
        if not script.__name__.startswith(required_prefix):
            raise ValueError("Script name (" + script.__name__ + \
                             ") does not start with required prefix " + \
                             required_prefix)
        scriptname = script.__name__.split(".")[-1] + '.py'
        try:
            save_argv = sys.argv
            save_stderr = sys.stderr
            sys.stderr = sys.stdout
            sys.argv = [scriptname] + args
            script.main()
        finally:
            sys.stderr = save_stderr
            sys.argv = save_argv

    def get_modpipe_binary_path(self, binary):
        """Get the full path to a ModPipe binary (e.g. main/BenchMark.pl)."""
        return os.path.join(modpipe.binaries._modpipebase, binary)

    def run_binary(self, binary, args):
        """Run a binary and require it to exit with status 0."""
        p = subprocess.Popen([binary] + args)
        self.require_clean_exit(p)

    def require_clean_exit(self, popen):
        """Raise an error if a subprocess exits with a non-zero return code."""
        ret = popen.wait()
        if ret != 0:
            raise OSError("External program returned exit code %d" % ret)

    def run_perl_code(self, code):
        """Run a short snippet of Perl code"""
        runscript = os.path.join(modpipe.binaries._modpipebase, 'int',
                                 'runscript.sh')
        p = subprocess.Popen([runscript, '/usr/bin/perl', '-w'],
                             stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                             stderr=subprocess.STDOUT,
                             universal_newlines=True)
        p.stdin.write(code)
        p.stdin.close()
        sys.stdout.write(p.stdout.read())
        p.stdout.close()
        self.require_clean_exit(p)
