#!/usr/bin/perl
# This file is part of ModPipe, Copyright 1997-2014 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

use Getopt::Long;
use IO::File;
use File::Basename;
use strict;

# --- Load local modules
use MPLib::Version;
use MPLib::Binaries;
use PLLib::Utils;
use PLLib::Sequence;
use PLLib::Modeller;
use PLLib::ModProfile;
use PLLib::PDBUtils;
use PLLib::ModPy;

my $default_pdb_repository = GetPDBRepository(0);

{
# --- Process command line options

$Getopt::Long::ignorecase = 0;

my ( $pssmdbname, $pssmdbfmt, $prflist, 
     $tgtprf, $evcut, $scrfile, $ccmatoffset,
     $prfformat, $scrstat, $rrfile,
     $matoffset, $gap_o, $gap_e, $pssmwght,
     $runname, $help, $usage, $tgtcode,
     $alidir, $hitfile, $alnbase,
     $summfile, $assumepdb, $pdbrep,
     $cleanup, $natpdb, $natchn, $nattyp,
     $customtag, $scoretype ) = undef;

GetOptions (
           "profile_pssmdb=s"           =>      \$pssmdbname,
           "pssmdb_format=s"            =>      \$pssmdbfmt,
           "profile_list=s"		=>      \$prflist,
           "target_profile=s"		=>      \$tgtprf,
           "profile_format=s"		=>      \$prfformat,
           "e_value=f"			=>      \$evcut,
           "output_scorefile=s"		=>      \$scrfile,
           "set_score_statistics=s"	=>      \$scrstat,
           "runname=s"			=>      \$runname,
           "substitution_matrix=s"	=>      \$rrfile,
           "matrix_offset=i"		=>      \$matoffset,
           "ccmatrix_offset=i"		=>      \$ccmatoffset,
           "gap_open_cost=i"		=>      \$gap_o,
           "gap_extend_cost=i"		=>      \$gap_e,
           "pssm_weighting_scheme=s"	=>      \$pssmwght,
           "target_code=s"		=>	\$tgtcode,
           "native_pdb=s"		=>	\$natpdb,
           "native_chn:s"		=>	\$natchn,
           "native_type=s"		=>	\$nattyp,
           "output_alidir=s"		=>	\$alidir,
           "hitfile_name=s"		=>	\$hitfile,
           "alignment_basename=s"	=>	\$alnbase,
           "custom_tag=s"               =>      \$customtag,
           "summary_file=s"		=>	\$summfile,
           "set_assume_pdb=s"		=>	\$assumepdb,
           "pdb_repository=s"		=>	\$pdbrep,
           "score_type=s"               =>      \$scoretype,
           "clean_up=s"			=>	\$cleanup,
           "help"			=>      \$help,
           "version"                    => sub { VersionMessage() },
           );

# --- Print help
if ( $help ){
   &usage;
   exit 0;
}

# --- Get Program name
my $subrname = GetSubrName();

# --- Check command-line options
unless ( $tgtprf && $tgtcode ){
   warn "${subrname}__E> Specify the input profile and target code\n";
   die "Try $subrname --help for usage information\n";
}

# --- Set various default parameters
$pssmdbname = $pssmdbname ? $pssmdbname : "pdb95-profiles.pssm";
$pssmdbfmt  = $pssmdbfmt ? $pssmdbfmt : "TEXT";
$prflist    = $prflist ? $prflist : 'pdb95-profiles.list';
$evcut      = $evcut ? $evcut : 1.0;
$runname    = "${tgtcode}-prfprf" unless ( $runname );
$rrfile     = 'blosum62.sim.mat' unless ( $rrfile );

$ccmatoffset = $ccmatoffset ? $ccmatoffset : -100;
$matoffset   = $matoffset ? $matoffset : -450;
$gap_o       = $gap_o ? $gap_o : -700; 
$gap_e       = $gap_e ? $gap_e : -70; 

$pssmwght  = 'HH1' unless ( $pssmwght );
$alidir    = "${tgtcode}-prfprf-ali" unless ( $alidir );
$hitfile   = "${tgtcode}-prfprf.hits" unless ( $hitfile );
$alnbase   = "${tgtcode}-prfprf" unless ( $alnbase );
$prfformat = 'PROFILE' unless ( $prfformat );
$summfile  = "${tgtcode}-prfprf.sum" unless ( $summfile );
$pdbrep    = $default_pdb_repository unless ($pdbrep);
$customtag = "PPSCAN (Prf-Prf)" unless ( $customtag );
$scoretype = 'CCMAT' unless ( $scoretype );

$scrstat   = ( $scrstat =~ /OFF/i ) ? 'False' : 'True';
$assumepdb = 'ON' unless ( $assumepdb );
$cleanup   = 'ON' unless ( $cleanup );

$natpdb     = undef unless ( $natpdb );
$natchn   = ' ' if ( ! $natchn || $natchn eq '' || $natchn eq '_' );
$nattyp     = 'sequence' unless ( $nattyp =~ /^structure$/i);

# -- Make sure that natpdb is specified if nattyp eq structure
if ( $nattyp eq 'structure' && ! defined($natpdb) ){
  warn "${subrname}__E> Native PDB code should be specified if target\n";
  warn "${subrname}__E> sequence type is STRUCTURE\n";
  die  "Try $subrname --help for usage information\n";
}

# --- Check for database files
die "${subrname}__E> Could not find list of database profiles: $prflist\n"
   unless ( -e $prflist );
die "${subrname}__E> Could not find PSSM database: $pssmdbname\n"
   unless ( -e $pssmdbname );

# --- Check for input files
die "${subrname}__E> Could not find input profile : $tgtprf\n"
   unless ( -e $tgtprf );

# --- Check target code in profile
die "${subrname}__E> Could not find the align code in input file: $tgtcode\n"
   unless ( CheckAlignCode($tgtprf, $prfformat, $tgtcode) );

# --- Initialize the cleanup array
my @cleanup_files = ();

# --- Create the python file variables
my $pynam  = "${runname}.py";
my $lognam = "${runname}.log";

# --- Open the python file
my $pyfh = OpenNewFile( $pynam );

# --- The conventions for profile types are different in MODELLER
#     and ModPipe. Hence the following switch is required. PPSCAN
#     (in MODELLER) thinks profiles can be either TEXT or BINARY
#     but MODPIPE thinks profiles can be either PIR, FASTA or
#     PROFILE.
$prfformat = 'TEXT' if ( $prfformat =~ /\bPROFILE\b/i );

# --- Write the python file for ppscan
WRTPY_PPSCAN($pyfh, $tgtprf, $prfformat,
             $pssmdbname, $pssmdbfmt, $prflist,
             $matoffset, $ccmatoffset, $rrfile,
             $gap_o, $gap_e, $scrstat, 'True',
             $scrfile, $pssmwght, $evcut, $alnbase,
             $summfile, $scoretype);

# --- Close python file
close( $pyfh );

# --- Run MODELLER for PPSCAN
my $starttime = time();
RunModeller($pynam)
   or die "${subrname}__E> MODELLER failed on profile: $tgtprf\n";
my $runtime = time() - $starttime;

# -- Check if log file exists
(-e $lognam) or
   die "${subrname}__E> Could not find MODELLER log file: $lognam\n";

# --- Check if output file exists
(-e $summfile) or
   die "${subrname}__E> PPSCAN (Prf-Prf) output not found: $summfile\n";

# -- Parse the log file to get the quality of the statistics

  # -- Open the log file
  my $fh_log = IO::File->new("< $lognam");

  # -- Check divergence, iterations etc
  my ($iter, $chi2low, $chi2high, $kstatlow, $kstathigh, $div)
        = ProfileStatInfo($fh_log);

  close($fh_log);

  # -- Print out the statistics
  printf "${subrname}__M> MODELLER Runtime: %8.2f minutes\n", $runtime / 60.0;
  print "${subrname}__M> No. of iterations: $iter\n";
  print "${subrname}__M> Chi2 fit of distributions: $chi2low\n";
  print "${subrname}__M> KS-Stat for the distributions: $kstatlow\n";

  warn "${subrname}__W> Fit b/w observed & expected distributions is bad\n"
    if ( $div );

# --- Update the cleanup list
push @cleanup_files, $pynam, $lognam, $scrfile, $summfile;

# --- Open the stream for the output file
my $fh_sum = OpenFile( $summfile ) or
   die "${subrname}__E> Failed opening file: $summfile\n";

# --- Verify if all alignments produced by PPSCAN exist
my $alicnt = CheckSummaryPPS($fh_sum);
#   die "${subrname}__E> Could not find all the alignment files produced by PPSCAN\n";

# --- Stop here if there are no hits
if ( $alicnt < 1 ){
  print "${subrname}__W> No hits against template database\n";
  print "${subrname}__W>    OUTPUT FILE: $summfile\n";
  exit 0;
}

# --- Report the number of alignment files found
warn "${subrname}__M> No. of hits found below E-value threshold: $alicnt\n";

# --- Open the filehandle for hit reports
my $fh_hit = OpenNewFile( $hitfile );

# --- Process each alignment produced
my @garbage = ProcessAliPPS($fh_sum, $tgtcode, $natpdb, $natchn, $nattyp, 
                            $assumepdb, $pdbrep, $alidir, 
                            $customtag, $fh_hit, $chi2low, $kstatlow);
push @cleanup_files, @garbage;

# --- Close summary file
close($fh_sum);

# --- Close the hit file
close($fh_hit);

# --- Clean up files
if ( $cleanup =~ /ON/i ){

   foreach my $file ( @cleanup_files ){
      unlink( $file ) if ( -e $file );
   }
}

# --- Exit finally
exit 0;
}

# --- Process Alignments
sub ProcessAliPPS {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 12;

   unless (scalar(@_) == $nargs){
      print "${subname}__D> Insufficient arguments\n" ;
      return;
   }

   # --- reassign input arguments
   my ($fh_summ, $tgtcode, $natpdb, $natchn, $nattyp, $assumepdb, $pdbrep, 
       $alidir, $customtag, $fh_hit, $chi2low, $kstatlow) = @_;

   # --- reset file position
   local *FH_SUMM = $fh_summ;
   seek(FH_SUMM,0,0);

   # --- Initialize array for garbage removal
   my @cleanup_files = ();

   # --- Start reading through the filehandle
   while ( my $sumln = <FH_SUMM> ){
      chomp $sumln;

      # --- Parse out the fields from the summary line
      my ($tgt, $tgtlen, $tgtbeg, $tgtend,
          $tmp, $tmplen, $tmpbeg, $tmpend,
          $neqv, $alnscore, $seqid, $zscore,
          $evalue, $alnfile) = split(" ", $sumln);

      # --- Check if the aln file exists (double check)
      unless ( -e $alnfile ){
         warn "${subname}__E> Could not find alignment file: $alnfile\n";
         next;
      }

      # --- Check if target code is present in file
      unless ( CheckAlignCode($alnfile, 'PROFILE', $tgtcode) ){
         warn "${subname}__E> Could not find target in alignment\n";
         warn "       Target   : $tgtcode\n";
         warn "       Alignment: $alnfile\n";
         next;
      }

      # --- Open the stream for the alignment file
      my $fh_aln = OpenFile( $alnfile ) or
         die "${subname}__E> Failed opening file: $alnfile\n";

      # --- Read the alignment that is profile format
      my ($target, $templates) = ReadProfileLines($fh_aln, $tgtcode);

      # --- Close the alignment file
      close($fh_aln);

      # --- Report number of hits
      my $numhits = scalar ( @$templates );
      print "${subname}__M> Processing alignment file (Prf-Prf or Seq-Prf) : $alnfile\n";
      print "${subname}__M>     Input target code               : $tgtcode\n";
      print "${subname}__M>     Number of templates in alignment: $numhits\n";

      # --- Get the range of the profile alignment
      my ($alibeg, $aliend) = GetRangePrfAli($target, $templates);

      # --- Format the target sequence (aligned part) in PIR
      # --- Get the target code
      my ($tgt_code, $tgt_length, $tgt_start, 
          $tgt_stop, $tgt_seqid, $tgt_evalue, 
          $tgt_seq ) = (split(" ", $target))[1,4,5,6,10,11,12];

      # -- Specify few other fields for the PIR format
      my $tgt_chn = 'A';          # -- Will be changed if target is structure.
      my $tgt_typ = 'sequence';   # -- Will be changed if target is structure.
      my $tgt_pdb = $tgt_code;    # -- Will be changed if target is structure.

      # --- Extract the target substring
      my $tgtseq = substr($tgt_seq, $alibeg, $aliend - $alibeg);

      # --- Find the region of the target covered by the alignment
      my $tgt_seq_copy = RemoveDashes($tgt_seq);
      my $tgtseq_copy  = RemoveDashes($tgtseq);
      my $tgt_length   = length($tgt_seq_copy);
      unless (($tgt_start, $tgt_stop) = 
                  FindSeqInSeq($tgtseq_copy, $tgt_seq_copy)){
         warn "${subname}__W> Failed fetching bounds for target: $tgtcode\n";
         $tgt_start = $tgt_stop = 0;
      }

      # -- If the target sequence is from a PDB, get the PDB
      #    numbering.
      if ( defined( $natpdb ) && $assumepdb =~ /ON/i ){
         # -- Get the beg/end of the PDB chain
         if (($tgt_start, $tgt_stop) =
                GetPDBResNum($tgtseq, $natpdb, $natchn, $pdbrep)){
           $tgt_chn = $natchn;
           unless ( $nattyp eq 'sequence' ){
             $tgt_typ = 'structureX';
             $tgt_pdb = $natpdb;
           }
         } else {
             warn "${subname}__W> Failed fetching bounds for native PDB\n";
             warn "${subname}__W>    Native PDB code: $natpdb\n";
             $tgt_start = $tgt_stop = 0;
           }
      }

      #-- Write output in PIR format (target)                                                                              
      # --- No need to confirm the coverage of the target sequence
      #     since MODELLER will do that by default
      my $tgtpir = WritePIR($tgt_code, $tgt_typ, $tgt_pdb,
                            $tgt_start, $tgt_chn, $tgt_stop, $tgt_chn,
                            $tgt_code, $customtag,
                            -1, -1, $tgtseq, 70);

      # --- Format each of the template sequences
      my $tmppir = undef;
      foreach my $template ( @$templates ){

         # --- Get the target code
         my ($tmp_code, $tmp_length, $tmp_start, 
             $tmp_stop, $tmp_seqid, $tmp_evalue, 
             $tmp_seq ) = (split(" ", $template))[1,4,7,8,10,11,12];

         # --- Extract the template substring
         my $tmpseq = substr($tmp_seq, $alibeg, $aliend - $alibeg);

         # --- Assume a pdb template and parse the ATOM_FILE and CHN 
         #     records. There is no automatic solution otherwise. 
         my ( $tmp_pdb, $tmp_chn ) = undef;
         if ( $assumepdb =~ /ON/i ){

            $tmp_pdb = substr($tmp_code, 0, 4);
            $tmp_chn = substr($tmp_code, 4, 1);
            $tmp_start = '.';
            $tmp_stop  = '.';

            # --- Create & locate the pdb filename
            my $pdbf = undef;
            unless ($pdbf = CheckPDBFile($tmp_pdb, $pdbrep)){
               warn "${subname}__E> Could not find PDB file   : $tmp_pdb\n";
               warn "${subname}__E> Specified PDB repositories: $pdbrep\n";
               next;
            }

            # --- Create the pdb filename
            my $pdbent = basename($pdbf);

            # --- Copy the pdb file locally
            unless ( CopyFile(${pdbf}, ${pdbent}) ){
               warn "${subname}__E> Could not find coordinate file\n";
               warn "               PDB repository  : $pdbrep\n";
               warn "               Atom file record: $tmp_pdb\n";
               next;
            }

            # --- Add pdb file to garabage list
            push(@cleanup_files, $pdbent);

         } else {
               $tmp_pdb = ' ';
               $tmp_chn = ' ';
           }

         # --- Write output in PIR format (template)
         my $seqtmp = WritePIR($tmp_code, 'structureX', $tmp_pdb,
                               $tmp_start, $tmp_chn, $tmp_stop, $tmp_chn,
                               $tmp_code, $customtag,
                               -1, -1, $tmpseq, 70);

         # --- Append seq to template sequence string
         $tmppir .= $seqtmp;
      }

      # --- Once you are here, you should have the target & template(s)
      #     in PIR format. Write it to a temporary file.
      my $temptag = TMPName();
      my $tempali  = "${tgtcode}-${temptag}.ali";

      # --- Write the alignment to the temporary file
      my $fh_tempali = OpenNewFile( $tempali );
      print $fh_tempali $tgtpir;
      print $fh_tempali $tmppir;
      close ($fh_tempali);
      
      # --- Update cleanup files
      push @cleanup_files, $tempali;

      # --- Verify the PDB numbering in the alignment
      #     Only if assumepdb is ON

      # --- Create names for rwali step
      my $temprwtag = "${tgtcode}-${temptag}-rwali";
      my $temprwtop = "${tgtcode}-${temptag}-rwali.py";
      my $temprwlog = "${tgtcode}-${temptag}-rwali.log";
      my $temprwali = "${tgtcode}-${temptag}-rwali.ali";

      # --- Open the Python file (RWALI)
      my $fhtop = OpenNewFile( $temprwtop );

      # --- Write the Python file for RWALI
      WRTTOP_RWALI($fhtop, $pdbrep, $tempali, 'PIR',
                   $temprwali, 'PIR');

      # --- Close Python file (RWALI)
      close( $fhtop );

      # --- Run MODELLER for RWALI
      unless ( RunModeller($temprwtop) ){
         warn "${subname}__E> MODELLER failed on rwali: $tgtcode\n";
         warn "${subname}__E> Failed processing alignment for modeling:\n";
         warn "               Alignment file: $alnfile\n";
         open ("F","$alnfile") or die ("can't open Alignment file $alnfile\n");
         while (my $line=<F>) {
             warn "               $line";
         }
         next;
      }

      # --- Make sure MODELLER wrote the final alignment
      unless ( -e $temprwali ){
         warn "Could not find output of MODELLER: $temprwali\n";
         next;
      }

      # --- Update garbage list
      push @cleanup_files, $temprwtop, $temprwlog, $temprwali;

      # --- If you are here, you should have an alignment that could
      #     be recognized by MODELLER for modeling. Safe to proceed
      #     onwards.

      my @seqids = ();
      unless (@seqids = SeqIdFromAli( $temprwali, 'PIR', 'ON', $pdbrep )){
         warn "${subname}__E> Failed calculating sequence identities\n";
         warn "${subname}__E>    Alignment file: $temprwali\n";
         next;
      }

      # --- Shift it one position to remove target
      shift @seqids;

      # --- Find the largest seqid
      my @seqid_c = sort { $b <=> $a } @seqids;
      my $hseqid = shift @seqid_c;

      # --- Initialize template string
      my $template_str = sprintf "%5d|", $hseqid;

      # --- Parse the alignment file one last time to get template details
      my $fh_pir = OpenFile($temprwali); my $template_cnt = 0;
      while ( defined(my $seq = ReadNextSeq($fh_pir, 'PIR')) ){

         my $prottyp = (ParseSeq($seq, 'PIR','OFF'))[1];

         # -- This correction is needed when the target sequence from PDB.
         #    Otherwise, the output string is screwed up. It assumes that
         #    the first sequence is the target - which is okay since it
         #    is written out that way by this same script.
         if ( defined($natpdb) &&
              $prottyp =~ /structure/i && 
              $template_cnt == 0 ){
           $template_cnt++;
           next;
         }

         if ( $prottyp =~ /structure/i ){
            # --- Increment the template count
            $template_cnt++;

            # --- Parse the  seq information
            my ($idcode, $pdbbeg, $pdbend) 
               = (ParseSeq($seq, 'PIR','OFF'))[0,3,5];

            # --- Format output string
            my $seqid = shift @seqids;
            $template_str .= 
               sprintf("%-5s %5d %5d %5d;", $idcode, $pdbbeg, $pdbend, $seqid);
         }
      }

      # --- Go through the alignment and get the percentage of gaps
      #     For a multiple sequence alignment, this will compare all
      #     pairwise alignments and report the lowest value of the 
      #     ratio of gaps to aligned positions with respect to the
      #     reference sequence.
      my ($perc_gaps, $max_seqid) = GetPercGaps($temprwali, 'PIR', $tgtcode);

      # --- Create output alignment directory
      MakeDirSys($alidir);

      # --- Move the alignement in to the new (final) directory
      my $alifile = "${alidir}/${alnfile}";
      unless ( CopyFile($temprwali, $alifile) ){
         warn "${subname}__E> Could not copy alignment to final destination\n";
         warn "               Source: $temprwali\n";
         warn "               Target: $alifile\n";
      }

      # --- Report progress
      printf "PRFPRF: %40s %5d %5d %5d|%40s|%8.4f|%8.4f|%10.2g|%3d|%5d %s\n", 
             $tgt, $tgt_length, $tgt_start, $tgt_stop, 
             $tmp, $chi2low, $kstatlow, $evalue, $perc_gaps, $template_cnt, 
             $template_str;

      # --- Report HIT
      printf $fh_hit "PRFPRF: %40s|%5d|%5d|%5d|%40s|%8.4f|%8.4f|%3d|%10.2g|%5d|%s #%s\n", 
                     $tgt, $tgt_length, $tgt_start, $tgt_stop, 
                     $tmp, $chi2low, $kstatlow, $evalue, $perc_gaps, $template_cnt, 
                     $template_str,
                     $alifile;


      # --- Nullify the references
      ($templates) = undef;
   }

   # --- Return the files created
   return @cleanup_files;

}




# --- Usage
sub usage {
print <<EOF;
${0}:

      --profile_list		File containing the names of the database
				profiles. Same as the PROFILE_LIST_FILE of
				PPSCAN. Default: profiles.list

      --profile_pssmdb		Name of the PSSM database, created using
                                make_pssmdb in MODELLER. 
                                Default: pdb95-profiles.pssm

      --pssmdb_format		Format of the PSSM database. Currently,
                                only the TEXT version is implemented.
                                Default: TEXT

      --target_profile		File containing the target profile

      --profile_format		Format of the profiles. Can be PROFILE or
                                PIR. Default: PROFILE

      --score_type		Type of scoring to use when calculating
                                alignments. Possible values are: CCMAT,
                                SEQPRF, CONS_BY_MAXPSSM, CONS_BY_MAXFREQ,
                                CONS_BY_MAXSUBS. Default: CCMAT

      --e_value			E-Value threshold to use to report hits
				against the database. Default: 1.0

      --output_scorefile	If specified, scores between target and all
				database profiles get stored in this file
				Default: <target_code>-prfprf.scores

      --set_score_statistics	Flag to trigger the calculation of E-values.
				Set it to OFF when there are only a few
				profiles in the database. Default: ON

      --runname			Run name. This will be used to name the
				MODELLER Python file.
				Default: <target_code>-prfprf

      --substitution_matrix	Substitution matrix to use. This file
				be available in the \${LIB}/ directory
				of MODELLER. Default: blosum62.sim.mat

      --matrix_offset		Value used to offset the alignment matrix.
				Best left at default value. Default: -450

      --ccmatrix_offset		Value used to offset the correlation 
                                coefficient matrix. Best left at default 
                                value. Default: -100

      --gap_open_cost		Cost to open a gap. Best left at the
				default. Default: -700

      --gap_extend_cost		Cost to extend a gap. Best left at the
				default value. Default: -70

      --pssm_weighting_scheme	Weighting scheme to use in the calculation
				of the PSSM. Only two different implementations
				of the Henikoff & Henikkoff scheme are
				currently available - HH0 or HH1. Default: HH1

      --target_code		Code of the target sequence in the profile. It
				will not proceed if this is missing or wrong.

      --native_pdb              The PDB code of the target sequence. A convenience
                                feature that can be used when benchmarking with 
                                PDB sequences. The input should be in PDB 4-letter
                                format => 1abc. Specify the chain id using the
                                option below.

      --native_chn              Chain id of the target PDB.

      --native_type             Type for target sequence in the PIR alignment. Again a
                                convenience feature when benchmarking. Set it to
                                'sequence' (default) or 'structure', according to
                                whether you want to use the resulting alignment for
                                model building or some structure calculation resp.

      --output_alidir		Directory to store the output alignments.
				Default: <target_code>-prfprf-ali

      --hitfile_name		File to store the details of the hits.
				Default: <target_code>-prfprf.hits

      --alignment_basename	Basename for the output alignments. The
				actual alignment files will be based on this
				variable appended with a serial number.
                                Default: <target_code>-prfprf

      --custom_tag              This tag will be written out in the alignment file
                                in the ninth column of the output PIR files. This is
                                useful to identify all the alignments from a
                                particular run.
                                Default: PPSCAN (Prf-Prf)

      --summary_file		File to store the summary output from
				MODELLER. Default: <target_code>-prfprf.sum

      --set_assume_pdb		Flag to assume the database profiles are of PDB
                                sequences. In which case, the program will take
                                the profile codes to be the standard 5-letter
                                PDB representation and will write out modeling
                                alignments. Default: ON

      --pdb_repository		The PDB repositories to look for coordinate files.
                                Should exist if --set_assume_pdb = ON.
                                Default: $default_pdb_repository

      --clean_up		Clean up all the files that this program produced
                                except the final alignments. Default: ON

      --version                 Report version number of this program.

      --help			This help. Pipe it through 'more' if it
				scrolls off the screen.
EOF
}
