#!/usr/bin/python
# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function
from optparse import OptionParser
import modpipe.fold_assignment
from modpipe.src.HitsSeqSeq import add_common_options, set_common_defaults
from modpipe.src.HitsSeqSeq import process_hits
import modpipe.version
import modeller
import time
import os
import sys

def get_options():
    parser = OptionParser(version=modpipe.version.message())

    parser.set_usage("""
 Find profile-sequence hits by scanning a target profile against a database.

 Usage: %prog [options] target_code profile database

 profile is a file containing a profile of the sequence 'target_code',
 and database the sequence database.

 Run `%prog -h` for help information
""")

    parser.add_option("--profile_format", dest="proffmt",
                      metavar="FORMAT",
                      type="choice", choices=('FASTA', 'PIR', 'PROFILE'),
                      help="""Format of the input sequence file
                              (FASTA, PIR, PROFILE, default PROFILE)""",
                      default="PROFILE")
    add_common_options(parser, 'prfseq', "BUILDP (Prf-Seq)")
    opts, args = parser.parse_args()

    if len(args) != 3:
        parser.error("You must specify the target code, profile " + \
                     "and database file names")
    target_code, prffile, database = args

    # Set defaults
    set_common_defaults(opts, target_code, parser, 'prfseq')

    return target_code, prffile, database, opts


def set_modeller_verbosity(verbose):
    if verbose == 0:
        modeller.log.none()
    elif verbose == 1:
        modeller.log.minimal()
    else:
        modeller.log.verbose()


def report_hits_summary(hits, target_code, prffile, database):
    numhits = len(hits)
    sub = "HitsPrfSeq.py"
    print(sub + "__M> Found %d hits for %s (Prf-Seq)" % (numhits, target_code))
    print(sub + "__M>       Input   : %s" % prffile)
    print(sub + "__M>       Database: %s" % database)

    # Stop here if there are no hits
    if numhits < 1:
        print(sub + "__W> No hits against template database")
        sys.exit(0)


def main():
    target_code, prffile, database, opts = get_options()
    set_modeller_verbosity(opts.verbose)
    env = modeller.Environ()
    env.io.atom_files_directory = opts.pdbrep
    s = modpipe.fold_assignment.ProfileSequence(env, database, opts)
    hits = s.search(target_code, prffile)
    report_hits_summary(hits, target_code, prffile, database)
    process_hits(env, hits, target_code, opts, 'PRFSEQ')


if __name__ == '__main__':
    main()
