#!/usr/bin/perl
# This file is part of ModPipe, Copyright 1997-2014 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

use strict;
use Getopt::Long;
use File::Basename;

use MPLib::Version;
use MPLib::Binaries;
use PLLib::Utils;
use PLLib::HHSuiteUtils;

my ( $hhsuite, $input, $database, $evalue,
    $outa3m, $iterations, $cpu, $outfile,
    $help, $verbosity );

# -- Get command line options
GetOptions (
           "hhsuite=s"        => \$hhsuite,
           "input=s"          => \$input,
           "database=s"       => \$database,
           "outa3m=s"         => \$outa3m,
           "outfile=s"        => \$outfile,
           "evalue=s"         => \$evalue,
           "iterations=s"     => \$iterations,
           "cpu=s"            => \$cpu,
           "help"             => \$help,
           );

# -- Print usage information
if ( $help ) {
   &usage;
   exit 0;
}

# --- Get Program name
my $subrname = GetSubrName();

# -- Check input arguments
unless ( $input && $database && $hhsuite ){
   print "${subrname}__E> Input should be at least hhprogram, query sequence and database file\n\n";
   die "Try $subrname --help for usage information\n";
}



# --- Check for existence of the input file
die "${subrname}__E> Could not find query sequence file: $input\n"
   unless ( -e $input );

# -- Check for the existence of blast databases
die "${subrname}__E> Could not find required HHsuite databases\n"
   unless ( CheckHHSuiteDB( $database, $hhsuite ) );

# -- Check if HHsuite executable exists
my ($hhlib, $hhsuiteexe) = GetHHSuite($hhsuite);
if (!defined $ENV{'HHLIB'}) {
   $ENV{'HHLIB'} = $hhlib;
}
die "${subrname}__E> Could not find HHsuite executable: $hhsuite, $hhsuiteexe\n"
   unless ( $hhsuiteexe );

# -- Prepare the options hash 
my %opt = (
            "-i"    => $input,
            "-d"    => $database,
            "-e"    => $evalue,
            "-E"    => $evalue,
            "-oa3m" => $outa3m,
            "-o"    => $outfile,
            "-cpu" => $cpu,
            );

# -- Run the HHprogram executable 
if ($hhsuite eq "hhblits") {
   $opt{"-n"} = $iterations;
}
system($hhsuiteexe, %opt);

exit 0;

# --- Usage
sub usage {
print <<EOF;
${0}:

      --hhsuite                 HHSuite program (hhsearch or hhblits)
      --input                   Input file. Should be a FASTA sequence
                                or a HHSuite .a3m file. Will not proceed without
                                this option.
      --database                Name of the hhsuite database to use for
                                scaning. Will not proceed without this
                                option.
      --outa3m                  Name of the output file that will contain
                                the assembled multiple sequence alignment.
                                Default: <basename-of-query>.a3m
      --outfile                 Name of the output file that will contain
                                the PDB alignments.
                                Default: <basename-of-query>.hhr
      --iterations              Number of iteration (for hhblits)
                                Default: [2]
      --evalue   	        E-value threshold for inclusion in profile.
                                Default: 1
      --cpu		        No of CPUs used. Default: 1
      --version                 Report version number of this program.
      --help                    This help.
EOF
}
