# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

package PLLib::DateUtils;
require Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw( GetMonthName GetMonthNum ConvertDateString IsDateEarlier);
             

use strict;
use File::Basename;
use PLLib::Utils;

sub IsDateEarlier {

   # -- Get name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 2;
   CheckArgs($subname, $nargs, scalar(@_));

   # -- Get the arguments
   my ($date, $refdate) = @_;

   return ( $date < $refdate ) ? 1 : 0;
}

sub ConvertDateString {

   # -- Get name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 2;
   CheckArgs($subname, $nargs, scalar(@_));

   # -- Get the arguments
   my ($input, $inpformat) = @_;

   my ( $day, $mon, $year) = '';
   if ( $inpformat =~ /\bDDMMMYY\b/i || 
        $inpformat =~ /\bDD-MMM-YY\b/i ){
    
      # Remove dashes if any
      $input =~ s/\-//g;

      $day = substr($input, 0, 2);

      # Convert alphbetical month to number
      $mon = substr($input, 2, 3);
      $mon = GetMonthNum($mon);

      # Convert year
      $year = substr($input, 5, 2);
      $year = ( $year =~ /^0/ ) ? 2000 + $year : 1900 + $year;
   }

   # Return date
   return $year . sprintf("%02d", $mon) . sprintf("%02d", $day);
   
}

sub GetMonthNum {

   # -- Get name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 1;
   CheckArgs($subname, $nargs, scalar(@_));

   # -- Get the arguments
   my $mmm = $_[0];

   # -- Check bounds
   return unless ( length($mmm) == 3 );

   # -- Set months in text format
   my @months = ( "JAN", "FEB", "MAR", "APR", "MAY", "JUN",
                  "JUL", "AUG", "SEP", "OCT", "NOV", "DEC" );

   # -- Fetch the month
   my $idx = IndexInList(\@months, $mmm);
   
   return $idx + 1;
}

sub GetMonthName {

   # -- Get name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 1;
   CheckArgs($subname, $nargs, scalar(@_));

   # -- Get the arguments
   my $mm = $_[0];

   # -- Check bounds
   return unless ( $mm > 0 && $mm <= 12 );

   # -- Set months in text format
   my @months = ( "JAN", "FEB", "MAR", "APR", "MAY", "JUN",
                  "JUL", "AUG", "SEP", "OCT", "NOV", "DEC" );

   # -- Fetch the month
   my $mon = $months[$mm - 1];

   return $mon;
}

