#!/usr/bin/python
# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function
import modeller
from optparse import OptionParser
import modpipe.version
import modpipe.pdbutils
import sys, os, re

def get_options():
    """Parse command line options"""
    parser = OptionParser(version=modpipe.version.message())

    # Set defaults
    parser.set_usage("""
 This script takes a file with a list of PDB codes (of the form
 1abcA) and extracts the relevant sequences from the PDB files
 to an alignment file.

 Usage: %prog [options]

 Run `%prog -h` for help information
 """)

    # Populate options list
    parser.add_option("-f", "--file",
                 dest="pdblist",
                 type='string',
                 help="""List containing names of PDB files to process.
                      This is a mandatory option. The program will
                      not proceed without this argument. The format of
                      this file should be pdb 4-letter code followed by
                      chain identifier (eg. 1abcA)""",
                 metavar="FILE", default=None)
    pdb = modpipe.pdbutils.get_pdb_repository(include_local=True)
    parser.add_option("-p", "--pdb_repository",
                      dest="pdbrep",
                      help="""Directories containing PDB files. The default
                              value is """ + str(pdb),
                      default=pdb, metavar="DIR")
    parser.add_option("-o", "--output_file",
                 dest="outfile",
                 help="""Name of output alignment FILE. This is a mandatory
                      option.""",
                 metavar="FILE", default=None)

    # Check mandatory options
    opts, args = parser.parse_args()

    if not opts.pdblist or not opts.outfile:
        parser.print_help()
        sys.exit(1)

    if not os.path.isfile(opts.pdblist):
        parser.error("""Cannot proceed without an input file""")
    return opts

def main():
    opts = get_options()

    # -- Open file containing list of PDB files
    f=open(opts.pdblist, "r")

    # -- Initialize some modeller stuff
    env = modeller.Environ()
    env.io.atom_files_directory = opts.pdbrep
    aln = modeller.Alignment(env)

    for pdb in f:
        # Extract 4-letter code
        pdb=pdb.rstrip('\n')
        pdb4=pdb[0:4]
        pdb1=pdb[4:5]

        # Extract chains from file
        try:
            mdl = modeller.Model(env, file=pdb4,
                                 model_segment=('FIRST:'+pdb1, 'LAST:'+pdb1))
            aln.append_model(mdl, align_codes=pdb, atom_files=pdb4)
        except:
            print("Modeller found an error in PDB: %s" % pdb)

    aln.write(file=opts.outfile, alignment_format='pir')

if __name__ == "__main__":
    main()
