#!/usr/bin/python
# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

from optparse import OptionParser
import modpipe.version
import modpipe.pdbutils
from shutil import rmtree
from modpipe.ce import CE
import sys, os, subprocess, tempfile, re
import modeller

def main():
    """Get and parse the command line options."""

    # Parse command line options
    parser = OptionParser(version=modpipe.version.message())

    # Set defaults
    parser.set_usage("""
 This script takes two PDB protein chains and aligns them using CE. The
 resulting alignment is then also cleaned up to make it Modeller-friendly.
 The various structural overlap numbers are also reported.

 Usage: %prog [options]

 Run `%prog -h` for help information
 """)

    parser.set_defaults(pdb1='',
                        chn1='',
                        pdb2='',
                        chn2='',
                        ceinpf='',
                        ceoutf='',
                        cealif='',
                        )

    # Populate options list
    parser.add_option("-p", "--pdb1",
                 dest="pdb1",
                 type='string',
                 help="""PDB code of the first structure.
                      This is a mandatory option.""",
                 metavar="PDB")
    parser.add_option("-c", "--chn1",
                 dest="chn1",
                 type='string',
                 help="""PDB chain identifier of the first structure.
                      If not specified it will take the first chain.
                      """,
                 metavar="CHN")
    parser.add_option("-q", "--pdb2",
                 dest="pdb2",
                 type='string',
                 help="""PDB code of the second structure.
                      This is a mandatory option.""",
                 metavar="PDB")
    parser.add_option("-d", "--chn2",
                 dest="chn2",
                 type='string',
                 help="""PDB chain identifier of the second structure.
                      If not specified it will take the first chain.
                      """,
                 metavar="CHN")
    pdb = modpipe.pdbutils.get_pdb_repository(include_local=True)
    parser.add_option("-x", "--pdb_repository",
                      dest="pdbrep", type='string',
                      help="""Directory containing PDB files. The default
                              value is """ + pdb, default=pdb,
                      metavar="DIR")
    parser.add_option("-i", "--ce_input_file",
                      dest="ceinpf", type='string',
                      help="""Input file that already contains a pre-calculated
                              CE alignment. If specified, this file will used
                              instead of running CE.""",
                      metavar="FILE")
    parser.add_option("-o", "--ce_output_file",
                 dest="ceoutf",
                 type='string',
                 help="""File to store the CE program output""",
                 metavar="FILE")
    parser.add_option("-s", "--ce_alignment_file",
                 dest="cealif",
                 type='string',
                 help="""File to store the CE alignment output,
                      modified for compatability with Modeller.""",
                 metavar="FILE")

    # Check mandatory options
    opts, args = parser.parse_args()

    if not opts.pdb1 or not opts.pdb2:
        parser.print_help()
        sys.exit(1)

    return opts, args

def calculate_ce_alignment(env, pdb1, chn1, pdb2, chn2,
                           ceinpf, ceoutf, cealif):
    """This does the actual work of calculating the CE alignment
    and parsing it. Also converts the CE output to a form that is
    compatible with Modeller."""

    # Create a CE alignment obj and use it
    myce = CE(env, pdb1, chn1, pdb2, chn2)

    # Either use the given CE alignment file or run CE
    # to get the alignment
    if ceinpf:
        myce.read_program_output(ceinpf)
    else:
        myce.get_alignment()
        if ceoutf:
            myce.write_program_output(ceoutf)

    # Parse CE output and get the alignment
    myce.parse_output()

    # Clean up the alignment
    myce.fix_alignment()

    # Write out alignment if requested
    if cealif:
        myce.write_alignment(file=cealif)

    # Create superpositions
    results = myce.calculate_superpositions()

    del(myce)
    return results


if __name__ == "__main__":

    # Parse command line options
    opts, args = main()

    # -- Initialize some modeller stuff
    modeller.log.minimal()
    env = modeller.Environ()
    env.io.atom_files_directory = opts.pdbrep

    # Call function to calculate CE alignments and parse them
    calculate_ce_alignment(env, opts.pdb1, opts.chn1,
                           opts.pdb2, opts.chn2, opts.ceinpf,
                           opts.ceoutf, opts.cealif)
