#!/usr/bin/python
# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function
from modeller import *
from optparse import OptionParser
from shutil import rmtree
from modpipe.tmalign import TMAlign
import modpipe.version
import modpipe.pdbutils
from modpipe.scripts.doTMAlign import calculate_tm_alignment
import sys, os, subprocess, tempfile, re

def main():
    """Get and parse the command line options."""

    # Parse command line options
    parser = OptionParser(version=modpipe.version.message())

    # Set defaults
    parser.set_usage("""
 This script is a batch version of doTMAlign.py. It takes a target PDB protein
 chain and a list of a set of chains, and iteratively aligns the target against
 each chain in the list using TMalign.

 Usage: %prog [options]

 Run `%prog -h` for help information
 """)

    parser.set_defaults(pdb1='',
                        chn1='',
                        listf='',
                        tmout='',
                        tmali='',
                        resf='',
                        )

    # Populate options list
    parser.add_option("-p", "--pdb1",
                 dest="pdb1",
                 type='string',
                 help="""PDB code of the first structure.
                      This is a mandatory option.""",
                 metavar="PDB")
    parser.add_option("-c", "--chn1",
                 dest="chn1",
                 type='string',
                 help="""PDB chain identifier of the first structure.
                      If not specified it will take the first chain.
                      """,
                 metavar="CHN")
    parser.add_option("-l", "--list",
                 dest="listf",
                 type='string',
                 help="""File containing list of chains to scan
                      against. The list should be in the format
                      <pdbcode> <chain-id>, one pair per line.
                      This is a mandatory option.""",
                 metavar="FILE")
    pdb = modpipe.pdbutils.get_pdb_repository(include_local=True)
    parser.add_option("-x", "--pdb_repository",
                      dest="pdbrep", type='string',
                      help="""Directories containing PDB files. The default
                              value is """ + pdb, default=pdb,
                      metavar="DIR")
    parser.add_option("-o", "--tm_output",
                 dest="tmout",
                 action="store_true",
                 help="""Flag to store the TMalign program output""")
    parser.add_option("-s", "--tm_alignment",
                 dest="tmali",
                 action="store_true",
                 help="""Flag to store the TMalign alignment output,
                      modified for compatability with Modeller.""")
    parser.add_option("-z", "--output_results_file",
                 dest="resf",
                 type='string',
                 help="""Filename for storing structure overlap results""",
                 metavar="FILE")

    # Check mandatory options
    opts, args = parser.parse_args()

    if not opts.pdb1 or not opts.listf:
        parser.print_help()
        sys.exit(1)

    if not os.path.isfile(opts.listf):
        parser.exit(1, 'List file '+opts.listf+' not found\n')

    return opts, args

if __name__ == "__main__":

    # Parse command line options
    opts, args = main()

    # Open file stream
    if opts.resf:
        o = open(opts.resf, 'w')
    else:
        o = sys.stdout

    # Process the list
    f = open(opts.listf, 'r')
    for line in f:
        li = line.split()
        if len(li) < 2:
            li.append('')

        tmoutf=''
        if opts.tmout:
            tmoutf = opts.pdb1 + opts.chn1 + '-' + li[0] + li[1] + '-tmalign.out'

        tmalif=''
        if opts.tmali:
            tmalif = opts.pdb1 + opts.chn1 + '-' + li[0] + li[1] + '-tmalign.ali'

        # Call function to calculate TMalign alignments and parse them
        try:
            results = calculate_tm_alignment(opts.pdbrep, opts.pdb1, opts.chn1,
                                          li[0], li[1], '', tmoutf, tmalif, (0,0,0,1,0))
        except:
            print("Failed calculating alignment b/w %s and %s" % \
                  (opts.pdb1+opts.chn1, li[0]+li[1]))
        o.write(results+'\n')
    f.close()
