from __future__ import print_function
import unittest
import modpipe.test
from modpipe.src import ConvertSeq
import os

class ConvertSeqTests(modpipe.test.TestCase):

    def _check_fasta(self, outfile, code, seq):
        with open(outfile) as fh:
            lines = fh.readlines()
        self.assertEqual(len(lines), 2)
        self.assertEqual(lines[0].rstrip(), '>' + code)
        self.assertEqual(lines[1].rstrip(), seq)
        os.unlink(outfile)

    def _check_pir(self, outfile, code, seq):
        with open(outfile) as fh:
            lines = fh.readlines()
        self.assertEqual(len(lines), 4)
        self.assertEqual(lines[0].rstrip(), '>P1;' + code)
        self.assertEqual(lines[1].rstrip(), 'sequence:::::::::')
        self.assertEqual(lines[2].rstrip(), seq)
        self.assertEqual(lines[3].rstrip(), '*')
        os.unlink(outfile)

    def _write_pir(self, infile, code, seq):
        with open(infile, 'w') as f:
            print("C; comment1", file=f)
            print("R; comment2", file=f)
            print(">P1;" + code, file=f)
            print("structureX:bar:1:A:106:A:bar:baz: 1.90: 0.19", file=f)
            print(seq + "*", file=f)

    def _write_fasta(self, infile, code, seq):
        with open(infile, 'w') as f:
            print(">" + code, file=f)
            print(seq, file=f)

    def test_pir_to_fasta(self):
        """Check conversion of PIR sequences to FASTA"""
        infile = 'infile'
        outfile = 'outfile'
        code = 'foo'
        for seq in ('MLGIIM', 'MLGiIm', 'MLGI IM'):
            self._write_pir(infile, code, seq)
            self.run_script('src', ConvertSeq,
                            ['-o', outfile, '-f', 'PIR', '-g', 'FASTA', infile])
            os.unlink(infile)
            self._check_fasta(outfile, code, 'MLGIIM')

    def test_fasta_to_pir(self):
        """Check conversion of FASTA sequences to PIR"""
        infile = 'infile'
        outfile = 'outfile'
        code = 'bar'
        for seq in ('CLGIIM', 'CLGiIm', 'CLGI IM'):
            self._write_fasta(infile, code, seq)
            self.run_script('src', ConvertSeq,
                            ['-o', outfile, '-f', 'FASTA', '-g', 'PIR', infile])
            os.unlink(infile)
            self._check_pir(outfile, code, 'CLGIIM')

if __name__ == '__main__':
    unittest.main()
