#include "bsp.h"


/******************************************************
 -------------     MUTATION    ------------------------
 ******************************************************
 Generate a random binding site with two properties:
  0 -> core
  1 -> surface
  2 -> binding site
*******************************************************/

// copy site1 to site2
void  site::copy( const site &old_site ){  
    ivec_cp(LENGTH, flag_patch, old_site.flag_patch);    
	update_patch();
	
    G      = old_site.G;
    for(int j=0; j<ENERGY_MAX; j++){
		GZ[j]  = old_site.GZ[j];
		GG[j]  = old_site.GG[j];
		GW[j]  = old_site.GW[j];
	}
  }

// measure the difference between site1 and site2
void  site::difference( const site &old_site, set *set ){  
	set->n = 0;
	for(int i =0; i<LENGTH; i++){
		if( flag_patch[i] != old_site.flag_patch[i]) {   
			set->v[set->n] = i;
			set->n++;
		}
  	}
  }

// Initialize site
void site::create_patch( prot_res *pr, int num_of_res)
{
  create_core(pr);

  int res_from_surf = surf.random();
  
  flag_patch[ res_from_surf ] = PATCH;
  update_patch();

  for(int j=1; j< num_of_res; j++){
    add_residue(pr);
  }
}

// Create core
void site::create_core( prot_res *pr)
{
  flag_patch = new int[LENGTH];
  core.init(LENGTH);
  surf.init(LENGTH);
  patch.init(LENGTH);

  for(int i=0; i<LENGTH; i++){    // distinguish between core and surface
    if( pr->res[i].nst.n == 0 ){
      flag_patch[i] = CORE;
    }
    else{
      flag_patch[i] = SURF;
    }
  }
  
  update_patch();
}


// Update patch - to be used every time a patch si modified
void site::update_patch( )
{
  core.determine_from_flag ( CORE,  flag_patch );
  surf.determine_from_flag ( SURF,  flag_patch );
  patch.determine_from_flag( PATCH, flag_patch );
}

/************************************************
  ----   MINIMIZATION   BY   MONTE   CARLO   ----
*************************************************/

void site::optimize( prot_res *pr)
{
  site site2;

  score( pr );
  
  site2.create_core(pr);
  site2.copy(*this);

  set muts;
  muts.v = new int[LENGTH];
  int nacc = 0;
  double temp = TEMP;
  int flag_acc = 0;

  for( int i=0 ; i < N_STEP ;  i++ ){
	temp *= TEMP_FCTR;
    site2.copy( *this );
  	site2.add_residue( pr ); 
  	//site2.remove_residue(  ); 
  	site2.remove_residue_nobreak( pr ); 
	site2.difference(*this, &muts);
    site2.score( pr );

    if( metrop( site2.G - G, temp)){
      copy( site2 );
	  flag_acc = 1;
      nacc++;
    }
	else{
		flag_acc = 0;
	}
	
	// mc1 debug
  	if(DEBUG>0){
  		fprintf(fp_mc1,"%d\t%d\t",nmc_counter, i);
		fprintf(fp_mc1,"%d\t%d\t", nacc,flag_acc);
		fprintf(fp_mc1,"%f\t", G);
		fprintf(fp_mc1,"%d\t%d\t%d\t", core.n,surf.n,patch.n);
  		fprintf(fp_mc1,"\n");
  	}
	// mc2 debug
  	if(DEBUG>0){
  		fprintf(fp_mc2,"%d\t", nmc_counter);
  		fprintf(fp_mc2,"%d\t", i);
  		fprintf(fp_mc2,"%f\t", temp);
  		fprintf(fp_mc2,"%f\t", G);
		for(int j=0; j<ENERGY_MAX; j++){
  			fprintf(fp_mc2,"%d-> %f %f\t",j,GG[j],GZ[j]);
		}
  		fprintf(fp_mc2,"\n");
  	}

	// mc3 debug
  	if(DEBUG>0){
  		fprintf(fp_mc3,"%d\t%d\t", nmc_counter, i);
		fprintf_flag(fp_mc3);
  	}
	// mc4 debug
  	if(DEBUG>0){
  		fprintf(fp_mc4,"%d\t%d\t", nmc_counter, i);
  		fprintf(fp_mc4,"%f\t%f\t", G, site2.G);
  		fprintf(fp_mc4,"%d\t", muts.n);
  		for(int j=0; j<muts.n;j++) fprintf(fp_mc4,"%d\t", muts.v[j]);
  		fprintf(fp_mc4,"\n");
  		fprintf(fp_mc4,"%d\t%d\t", nmc_counter, i);
		fprintf_flag(fp_mc4);
  		fprintf(fp_mc4,"%d\t%d\t", nmc_counter, i);
		site2.fprintf_flag(fp_mc4);
  	}
	
  }
}


//----ADD---RESIDUE---------------------

void site::add_residue( prot_res *pr )
{

  int flag[LENGTH];
  ivec_init(LENGTH, flag, 0);

  for(int i=0; i<LENGTH; i++){
    if( flag_patch[i] == PATCH ){
      for(int j=0; j< pr->res[i].nst.n; j++){
		int k = pr->res[i].nst.v[j];
		if( flag_patch[k] == SURF){
	  		flag[k] = 1;
		}
      }
    }
  }
  set set;
  set.init(LENGTH);
  set.determine_from_flag(1, flag);
  int res_to_add  = set.random();

  flag_patch[ res_to_add ] = PATCH;

  update_patch( );
}

//-----REMOVE----RESIDUE------------

void site::remove_residue( )
{
  int k = patch.random();

  flag_patch[k] = SURF;
  
  update_patch();

}

//-----REMOVE----RESIDUE------------

void site::remove_residue_nobreak(prot_res *pr )
{
	int nremove_step = 100;
	int destroy;
	
	for(int i=0; i<nremove_step; i++){
		// try residue
  		int k = patch.random();

  		// mutate residue 
		flag_patch[k] = SURF;
  		update_patch();
		
		//check if residue destroy patch
  		if( (destroy = destroy_patch(pr)) != 0){
  			flag_patch[k] = PATCH;
  			update_patch();
			//fprintf(stdout, "continue\n");
			//continue;
		}
		else{
			//fprintf(stdout, "return\n");
			return;
		}
	}

	fprintf(stderr, "ERROR in remove_residue\n");
	fprintf(stderr, "Couldn't remove residues in %d steps!\n", nremove_step);
	fflush(stderr);

	exit(1);
}


int site::destroy_patch(prot_res *pr){

	int flag[LENGTH];

	int r1 = patch.v[0];
	
	for(int i=0; i<LENGTH; i++){flag[i] =0;}
	
	flag[r1] = 1; //initialize seed
	
	int tot_count = 1;
	
	for(int iter=0; iter<100;iter++){
		int count=0;
		for(int i=0; i<LENGTH; i++){
			if(flag[i]==1){ //
				for(int j=0; j< pr->res[i].nst.n; j++){
					int k = pr->res[i].nst.v[j];
					if(flag[k]==0 && flag_patch[k]==PATCH){
						flag[k]=1;
						count++;
						tot_count++;
					}
				}
			}
		}
		//fprintf(stdout, "%d\t%d\t\n",iter,count);
		if(count==0){
			break;
		}
	}
	//fprintf(stdout, "return %d\t\n",tot_count-patch.n);
	
	return (tot_count-patch.n);
}

/**********************************************
-------OPERATIONS-----ON-------SET------------
**********************************************/
//Determine set
void set::determine_from_flag(int f, int flag[])
{
  n = 0;

  for(int i=0; i<LENGTH; i++){
    if( flag[i] == f ){
      v[n] = i;
      n++;
    }
  }
}

int inline set::random()
{
  int nr = (int)(ran3(&IDUM) * n);  

  return v[nr];
}
