/*******************************************************************
 *  BINDING SITE PREDICTION PROGRAM
 *
 *  Input: features (.cns,...)
 *  Output: files con binding sites (1,0)
 *
 *  Descritpion: this program
 *
 *
 *
 *
 ********************************************************************/

#include "/home/andrea/prot/C/lib/Include_basic.h"
//#include "Prot.h"
#include "atom.h"


int    SITE_SIZE = 10;
double PSA_SURF  = 10;
int    LOG = 0;
int    NMC_MAX = 1000;
double TEMP = 0.1;
double W1 = 10.0;
double W2 = 10.0;
double RMAX = 15.0;

int    LENGTH;


class prot Prot;

struct site{
  int n;
  int *flag;
  int *site;
  int *core;
  int *surf;
  double G;
  double G1;
  double G2;
} ;

struct cns{
  int *n1;
  char *c2;
  double *cns;
} Cns;


struct psa{
  int *numpdb;
  char **seq;
  double *side1;
  double *side2;
} Psa;

void write_site(char file[], struct site *site);


// Options
char    Xyz_file[100];     
char    Cns_file[100];
char    Psa_file[100];
char    Site_file[100];




double  Cutoff=6.0;      


char Usage[1000];

/*********************************************************************
 ****     MAIN    ****************************************************
 *********************************************************************/

int main(int argc, char *argv[])
{
  void getoptions(int argc, char *argv[]);
  void read_cns(char file[], struct cns *);
  void read_psa(char file[], struct psa *);
  void optimize_site(char file [], char file2[], char file3[]);

  getoptions(argc, argv);

  /* printf("\n");
     printf("Going to read xyz...\n");
     LENGTH = Prot.read_xyz(Xyz_file);
     Prot.create_IP('A');
     Prot.create_dmap();*/

  printf("Going to read sol...\n");
  Prot.read_sol( Xyz_file);
  Prot.write_sol("sol1");
  Prot.create_surfers(5.0);
  Prot.rasmol_surfers("surf2", 5.0);
  Prot.write_surfers("surf1");
  Prot.create_dmap();
  Prot.write_dmap("map1");

  exit(0);
  printf("Going to read cns...\n");  
  read_cns( Cns_file, &Cns );

  printf("Going to read psa...\n");
  read_psa( Psa_file, &Psa );

  printf("\n");
  printf("Optimization...\n");
  printf("\n");
  optimize_site("site1","site2","site3");
}

/******************************************************
-----------   Get Options  ----------------------
*******************************************************/
void getoptions(int argc, char *argv[])
{
  //-- Default values assigned
  Xyz_file[0]    = '\0';
  Cns_file[0]    = '\0'; 
  Psa_file[0]    = '\0'; 
  Site_file[0]    = '\0'; 
  
  // -- Usage
  strcpy (  Usage, "\n---\nOptions:\n");
  sprintf( Usage+strlen(Usage), "  -xyz     ->  Xyz file [no default, mandatory]\n");
  sprintf( Usage+strlen(Usage), "  -cns     ->  Cns file [no default]\n");
  sprintf( Usage+strlen(Usage), "  -psa     ->  Psa file [no default]\n");
  sprintf( Usage+strlen(Usage), "  -site    ->  Site file [no default]\n");
  sprintf( Usage+strlen(Usage), "  -idum    ->  Idum [default: -1]\n");


  // -- Reading
  for(int i=1; i<argc; i++){
    if      ( strcmp( "-xyz",    argv[i])==0 ) sscanf( argv[++i], "%s",   Xyz_file);
    else if ( strcmp( "-cns",    argv[i])==0 ) sscanf( argv[++i], "%s",   Cns_file);
    else if ( strcmp( "-psa",    argv[i])==0 ) sscanf( argv[++i], "%s",   Psa_file);
    else if ( strcmp( "-site",   argv[i])==0 ) sscanf( argv[++i], "%s",   Site_file);
    else if ( strcmp( "-idum",   argv[i])==0 ) sscanf( argv[++i], "%d",   &IDUM);
    else{
      printf( "Argv[%d] not recognized: <%s> = <%s>\n", i, argv[i], argv[i+1]);
      printf("%s\n", Usage);
      exit(0);
    }
  }

  // -- Mandatory
  if(Xyz_file[0]=='\0' ){
    printf("Xyz mandatory!!!\n");
    printf("%s\n", Usage);
    exit(0);
  }

  // -- Printing values
  printf("----------------------------------------------\n");
  printf("Here is the input:\n");
  printf("  Xyz_file   = %s\n", Xyz_file);
  printf("  Cns_file   = %s\n", Cns_file);
  printf("  Psa_file   = %s\n", Psa_file);
  printf("  Site_file  = %s\n", Site_file);
  printf("  IDUM       = %d\n", IDUM);
  printf("----------------------------------------------\n");
}


/*****************************************************
 -------------------    INPUT    ---------------------
 *****************************************************/
void read_cns(char file[], struct cns *cns)
{
  char pippo[1000];
  char buffer[1000];
  
  int n=0;

  cns->n1  = new int[LENGTH];
  cns->c2  = new char[LENGTH];
  cns->cns = new double[LENGTH];

  FILE *fp = fopen_read(Cns_file);
  for(int i=0 ;  ; i++){
    if(fgets(buffer, 1000, fp) == NULL){
      break;
    }
    //printf("%5d %s\n",i+1, buffer);
    if(buffer[0]!='#'){
      sscanf(buffer, "%d %c %lf", &cns->n1[n], &cns->c2[n], &cns->cns[n]);
      if(LOG==1) printf("%3d %3d %c %8.3f\n", n, cns->n1[n], cns->c2[n], cns->cns[n]);
      n++;
    }
  }

  if(n!=LENGTH){
    printf("ERRORE Length: %d n: %d\n", LENGTH, n);
    exit(0);
  }

  fclose(fp);
}

//Read PSA
void read_psa(char file[], struct psa *psa)
{
  char buffer[1000];

  psa->seq    = alloc_cmat(LENGTH,4);
  psa->numpdb = new int [LENGTH];
  psa->side1  = new double[LENGTH];
  psa->side2  = new double[LENGTH];

  int n=0;

  FILE *fp = fopen_read(file);
  for(int i=0; ; i++){
    if(fgets(buffer, 1000, fp) == NULL){
      break;
    }
    if(buffer[0]!='#'){
      sscanf(buffer+7, "%d%s", &psa->numpdb[n], &psa->seq[n][0]);
      //sscanf(buffer+60, "%lf%lf", &psa->side1[n], &psa->side2[n]);
      sscanf(buffer+20, "%lf%lf", &psa->side1[n], &psa->side2[n]);
      if(LOG==1) printf("%3d %3d %s %lf %lf\n", n+1, psa->numpdb[n], psa->seq[n], psa->side1[n], psa->side2[n]);
      n++;
    }
  }
  fclose(fp);
}

/************************************************
  ----   MINIMIZATION   BY   MONTE   CARLO   ----
*************************************************/
void create_site(struct site *site);
void create_core(struct site *site, struct psa *psa);
void init_site(struct site *site);
void add_residue(struct site *site);
void remove_residue(struct site *site);
int  determine_set(int f1, int flag[], int set[]);
void cp_site(struct site *site1, struct site *site2);
void mutate_site(struct site *site);
void score(struct site *site);
void check_site(struct site *site);
void rasmol_site(FILE *fp, struct site *site);
void fprintf_flag(FILE *fp, int flag[]);

void optimize_site(char file1[], char file2[], char file3[])
{
  struct site site1, site2;

  FILE *fp1 = fopen_write(file1);
  FILE *fp2 = fopen_write(file2);
  FILE *fp3 = fopen_write(file3);

  create_site(&site1);

  printf("  Create core...\n");
  create_core(&site1, &Psa);

  printf("  Initialize site...\n");
  init_site(&site1);

  score(&site1);

  create_site(&site2);

  write_site("pippo.dat", &site1);


  printf("  Starting minimization...\n\n");

  int nprop = 0;
  int nacc  = 0;

  for( int i=0 ; i < NMC_MAX ;  i++ ){
    cp_site( &site2, &site1 );
    mutate_site(&site2);
    score(&site2);

    if( metrop( site2.G - site1.G, TEMP)){
      cp_site( &site1, &site2 );
      nacc++;
    }

    fprintf(fp1,"%5d %5d  %6.2f  %6.2f %6.2f\n",i+1, nacc, site1.G, site1.G1, site1.G2);
    fprintf(fp2,"%5d ", i+1);
    fprintf_flag(fp2, site1.flag);
    fprintf(fp2, "\n");
    rasmol_site(fp3, &site1);
  }
  fclose(fp1);
  fclose(fp2);
}
/*****************************************************
 ****************    MUTATIONS   *********************
 *****************************************************/
// Generate a random binding site with two properties:
// 1) contiguity
// 2) non-core residues
//
//  0 -> core
//  1 -> surface
//  2 -> binding site
// Create site
void create_site(struct site *site)
{
  site->flag = new int[LENGTH];
}

// Copy site
void cp_site(struct site *site2, struct site *site1)
{
  for(int i=0; i<LENGTH; i++){
    site2->flag[i] = site1->flag[i];
  }
  site2->G  = site1->G;
  site2->G1 = site1->G1;
  site2->G2 = site1->G2;
}
// Create core
void create_core(struct site *site, struct psa *psa)
{
  for(int i=0; i<LENGTH; i++){    // distinguish between core and surface
    if( psa->side1[i] < PSA_SURF ){
      site->flag[i] = 0;
    }
    else{
      site->flag[i] = 1;
    }
  }
  int set[LENGTH];
  int nset = determine_set(0, site->flag, set);
  printf("    N. of core residues: %d\n", nset);
}

// Initialize site
void init_site(struct site *site)
{
  for(int j=0; j< SITE_SIZE; j++){
    add_residue(site);
  }
}

// Mutation
void mutate_site(struct site *site)
{
  add_residue(site);
  remove_residue(site);
}
// Add residue to binding site
void add_residue(struct site *site)
{
  int set[LENGTH];

  int nset = determine_set(1, site->flag, set);
  if(nset==0){
    printf("Error in add_residue: No set found!\n");
    check_site(site);
    exit(0);
  }

  int nr;
 
  nr = (int)(ran3(&IDUM)*nset);  // a random residue on the surface

  if ( site->flag[set[nr]] != 1){
    printf("Errore in add_residue!!\n");
    exit(0);
  }

  site->flag[set[nr]] = 2;
}

//Remove residue from binding site
void remove_residue(struct site *site)
{
  int set[LENGTH];

  int  nset = determine_set(2, site->flag, set);
  if(nset==0){
    printf("Error in remove_residue: No set found!\n");
    exit(0);
  }

  int nr = (int)(ran3(&IDUM)*nset);  // a random residue on the surface 
 
  if ( site->flag[set[nr]] != 2 ){
    printf("Errore in remove_residue!!\n");
    check_site(site);
    exit(0);
  }
  site->flag[set[nr]] = 1;
}

//Determine set
int determine_set(int f, int flag[], int set[])
{
  int nset = 0;
  for(int i=0; i<LENGTH; i++){
    if( flag[i] == f ){ 
      set[nset] = i;
      nset++;
    }
  }  
  //printf("nset %d\n", nset);
  return nset;
}

/*************************************************
---------------    CHECK AND WRITE     -----------
**************************************************/

// Check site
void check_site(struct site *site)
{
  int n0=0;
  int n1=0;
  int n2=0;;

  int set0[LENGTH];
  int set1[LENGTH];
  int set2[LENGTH];

  for(int i=0; i<LENGTH; i++){
    if     ( site->flag[i] == 0 ){ 
      set0[n0] = i+1;
      n0++;
    }
    else if( site->flag[i] == 1 ){ 
      set1[n1] = i+1;
      n1++;
    }
    else if( site->flag[i] == 2 ){ 
      set2[n2] = i+1;
      n2++;
    }
  }  

  printf(" 0 --> %3d || ", n1);
  for(int j=0; j<n0; j++){
    printf("%3d ", set0[j]);
  }
  printf("\n");

  printf(" 1 --> %3d || ", n1);
  for(int j=0; j<n1; j++){
    printf("%3d ", set1[j]);
  }
  printf("\n");

  printf(" 2 --> %3d || ", n1);
  for(int j=0; j<n2; j++){
    printf("%3d ", set2[j]);
  }
  printf("\n");
}
// Check site
void rasmol_site(FILE *fp, struct site *site)
{
  void fprintf_rasmol(FILE *fp, int n, int set[]);

  int n0=0;
  int n1=0;
  int n2=0;;

  int set0[LENGTH];
  int set1[LENGTH];
  int set2[LENGTH];

  for(int i=0; i<LENGTH; i++){
    if     ( site->flag[i] == 0 ){ 
      set0[n0] = Psa.numpdb[i];
      n0++;
    }
    else if( site->flag[i] == 1 ){ 
      set1[n1] = Psa.numpdb[i];
      n1++;
    }
    else if( site->flag[i] == 2 ){ 
      set2[n2] = Psa.numpdb[i];
      n2++;
    }
  }  

  fprintf_rasmol( fp, n0, set0);
  fprintf(fp, "color grey\n");

  fprintf_rasmol( fp, n1, set1);
  fprintf(fp, "color blue\n");

  fprintf_rasmol( fp, n2, set2);
  fprintf(fp, "color red\n");
}

// Print flag
void fprintf_rasmol(FILE *fp, int n, int set[])
{
  fprintf(fp, "select ");
  int j;
  for( j=0; j<n-1; j++){
    fprintf(fp, "%3d,", set[j]);
  }
  fprintf(fp, "%3d\n",set[j]);
}


//Write down binding site
void write_site(char file[], struct site *site)
{
  FILE *fp = fopen_write(file);
  fprintf(fp, "%d\n", LENGTH);
  for(int i=0; i< LENGTH; i++){
    fprintf(fp, "%3d %3d\n", i+1, site->flag[i]);
  }
  fclose(fp);
}

// Print flag
void fprintf_flag(FILE *fp, int flag[])
{
  for(int i=0; i<LENGTH; i++){
    fprintf(fp,"%1d", flag[i]);
  }
}

/***********************************************
 **   SCORE  FUNCTION
 ************************************************/
void score(struct site *site)
{
  double score_size(struct site *site);
  double score_cns(struct site *site);

  site->G1 = score_cns(site);
  site->G2 = score_size(site);

  site->G = (W1 * site->G1 + W2 * site->G2);
}

// From conservation a score for the binding site
// Using average
double score_cns(struct site *site)
{
  double a1 = 0.0;
  double a2 = 0.0;

  int n1 = 0;
  int n2 = 0;

  for(int i=0; i<LENGTH; i++){
    if( site->flag[i] == 1){    
      a1 += Cns.cns[i];
      n1++; 
    }
    else if(site->flag[i] == 2){ 
      a2 += Cns.cns[i];
      n2++;
    }
  }

  a1 /= (double) n1;
  a2 /= (double) n2;

  return a1-a2;
}

// Score the size of the patch
double score_size(struct site *site)
{
  double R = 0.0;

  for(int i =0; i< LENGTH; i++){
    if(site->flag[i] == 2){
      for(int j =0; j< LENGTH; j++){
	if(site->flag[j] == 2){
	  if(R < Prot.dmap[i][j]){
	    R = Prot.dmap[i][j];
	  }
	}
      }      
    }
  }

  return (R > RMAX)? R : RMAX;
}

// Contiguity
/*double score_ctg(struct site *site)
{
  int nctg = 0;

  for(int i =0; i< LENGTH; i++){
    if(site->flag[i] == 2){
      for(int j =0; j< LENGTH; j++){
	if(site->flag[j] == 2){
	  if(RCTG < Prot.dmap[i][j]){
	    nctg++;
	  }
	}
      }
    }
  }

  return  (double) nctg;
}

// Side
double score_side(struct site *site)
{
  int nctg = 0;

  for(int i =0; i< LENGTH; i++){
    if(site->flag[i] == 2){
      for(int j =0; j< LENGTH; j++){
	if(site->flag[j] == 2){
	  if(RCTG < Prot.dmap[i][j]){
	    nctg++;
	  }
	}
      }
    }
  }

  return  (double) nctg;
}
*/
